import {TRAIN_TARIFF_CLASSES} from 'projects/trains/lib/segments/tariffs/constants';
import {TRAIN_COACH_TYPE} from 'projects/trains/constants/coachType';

import {
    ETrainsFilterType,
    ITrainsFilters,
    ITrainsTrainTariffClassFilter,
} from 'types/trains/search/filters/ITrainsFilters';
import {ITrainsSearchLocation} from 'types/trains/search/ITrainsSearchLocation';
import {ITrainsTariffApiSegment} from 'server/api/TrainsApi/types/ITrainsGetTariffsApi/models';

import {getTariffClassKeys} from 'projects/trains/lib/segments/tariffs';
import filterTariffClassKeysByPriceRange from 'projects/trains/lib/filters/filterTariffClassKeysByPriceRange';
import BaseListManager from 'projects/trains/lib/filters/managers/utilities/BaseListManager';
import getTariffClass from 'projects/trains/lib/getTariffClass';

export class TrainTariffClass extends BaseListManager<
    TRAIN_COACH_TYPE,
    TRAIN_COACH_TYPE,
    ETrainsFilterType.TRAIN_TARIFF_CLASS
> {
    type: ETrainsFilterType.TRAIN_TARIFF_CLASS =
        ETrainsFilterType.TRAIN_TARIFF_CLASS;

    apply(
        value: TRAIN_COACH_TYPE[],
        segment: ITrainsTariffApiSegment,
    ): boolean {
        return getTariffClassKeys(segment).some(tariffClass =>
            value.includes(tariffClass),
        );
    }

    /**
     * Возвращает список доступных значений для данного фильтра,
     * с учетом результатов других фильтров.
     */
    getActiveOptions({
        filtersData,
        segments,
        filteredSegments = this.getFilteredSegments({segments, filtersData}),
    }: {
        filtersData: ITrainsFilters;
        segments: ITrainsTariffApiSegment[];
        filteredSegments?: ITrainsTariffApiSegment[];
    }): TRAIN_COACH_TYPE[] {
        const options = this.getOptions(segments);

        // проверяем какие опции соответствуют оставшимся сегментам
        return options.filter(option =>
            filteredSegments.some(segment => {
                let tariffClassKeys = getTariffClassKeys(segment);

                // список тарифов, подходящих под фильтр цены
                tariffClassKeys = filterTariffClassKeysByPriceRange(
                    tariffClassKeys,
                    segment,
                    filtersData,
                );

                return tariffClassKeys.includes(option);
            }),
        );
    }

    updateOptions(
        options: TRAIN_COACH_TYPE[],
        segment: ITrainsTariffApiSegment,
    ): TRAIN_COACH_TYPE[] {
        const segmentClasses = getTariffClassKeys(segment);

        if (
            segmentClasses.every(tariffClass => options.includes(tariffClass))
        ) {
            return options;
        }

        return TRAIN_TARIFF_CLASSES.filter(
            tariffClass =>
                options.includes(tariffClass) ||
                segmentClasses.includes(tariffClass),
        );
    }

    getOptions(segments: ITrainsTariffApiSegment[]): TRAIN_COACH_TYPE[] {
        return segments.reduce<TRAIN_COACH_TYPE[]>(
            (prevOptions, segment) => this.updateOptions(prevOptions, segment),
            [],
        );
    }

    /**
     * Возвращает список опций из данных обо всех фильтрах (из state).
     */
    getOptionsFromFiltersData(filtersData: ITrainsFilters): TRAIN_COACH_TYPE[] {
        return filtersData[this.type].options;
    }

    /**
     * Возвращает начальное состояние данного фильтра.
     */
    initFilterData(
        segments: ITrainsTariffApiSegment[],
    ): ITrainsTrainTariffClassFilter {
        const options = this.getOptions(segments);
        const availableWithOptions = this.isAvailableWithOptions(options);
        const filteredSegmentIndices: boolean[] = new Array(
            segments.length,
        ).fill(true);

        return {
            value: this.getDefaultValue(),
            options,
            activeOptions: options,
            availableWithOptions,
            availableWithActiveOptions: availableWithOptions,
            type: this.type,
            filteredSegmentIndices,
        };
    }

    deserializeFromQuery({
        trainTariffClass,
    }: ITrainsSearchLocation): TRAIN_COACH_TYPE[] {
        if (!trainTariffClass) {
            return this.getDefaultValue();
        }

        const queryClasses = Array.isArray(trainTariffClass)
            ? trainTariffClass
            : [trainTariffClass];

        return TRAIN_TARIFF_CLASSES.filter(tariffClass =>
            queryClasses.includes(tariffClass),
        );
    }

    serializeToQuery(value: TRAIN_COACH_TYPE[]): {
        trainTariffClass: TRAIN_COACH_TYPE[];
    } {
        return {trainTariffClass: value};
    }

    validateValue(
        value: TRAIN_COACH_TYPE[],
        options: TRAIN_COACH_TYPE[],
    ): TRAIN_COACH_TYPE[] {
        return value.filter(item => options.includes(item));
    }

    formatOptions(
        options: TRAIN_COACH_TYPE[],
        formatTextFn?: (
            option: {value: string; text: string},
            index: number,
        ) => string,
    ): {
        value: string;
        text: string;
    }[] {
        const formattedOptions: {value: string; text: string}[] = options.map(
            value => ({
                value,
                text: getTariffClass(value),
            }),
        );

        return formatTextFn
            ? this.formatTextWithFn(formattedOptions, formatTextFn)
            : formattedOptions;
    }
}

export default new TrainTariffClass();
