import {
    ETrainsFilterType,
    ITrainsFilters,
} from 'types/trains/search/filters/ITrainsFilters';
import {ITrainsTariffApiSegment} from 'server/api/TrainsApi/types/ITrainsGetTariffsApi/models';

import applyFilters from 'projects/trains/lib/filters/applyFilters';

export default class BaseListManager<
    Value,
    Option,
    Type extends ETrainsFilterType,
> {
    // @ts-ignore
    type: Type;

    apply(_value: Value[], _segment: ITrainsTariffApiSegment): boolean {
        return true;
    }

    /**
     * Проверяет должен ли применяться данный фильтр.
     *
     * @param value - текущее значение фильтра
     */
    shouldApply(value: Value[]): boolean {
        return !this.isDefaultValue(value);
    }

    getDefaultValue(): Value[] {
        return [];
    }

    isDefaultValue(value: Value[]): boolean {
        return value.length === 0;
    }

    /**
     * Производит промежуточное преобразование value для передачи его в отображение.
     * Ожидается, что метод вернет либо прмитивный тип, либо массив прмитивных типов.
     */
    formatValue(value: Value[]): Value[] {
        return value;
    }

    getDefaultOptions(): Option[] {
        return [];
    }

    updateOptions(
        options: Option[],
        _segment: ITrainsTariffApiSegment,
    ): Option[] {
        return options;
    }

    /**
     * Обновляет список опций на основе данных об отфильтрованных сегментах.
     */
    updateOptionsForFilteredSegments({
        options,
        filtersData: _filtersData,
        segments: _segments,
        filteredSegments: _filteredSegments,
    }: {
        options: Option[];
        filtersData: ITrainsFilters;
        segments: ITrainsTariffApiSegment[];
        filteredSegments: ITrainsTariffApiSegment[];
    }): Option[] {
        return options;
    }

    /**
     * Возвращает список сегментов, которые соответствую всем выбранным фильтрам без
     * учета данного фильтра.
     */
    getFilteredSegments({
        segments,
        filtersData,
    }: {
        segments: ITrainsTariffApiSegment[];
        filtersData: ITrainsFilters;
    }): ITrainsTariffApiSegment[] {
        return segments.reduce<ITrainsTariffApiSegment[]>(
            (filteredSegments, segment, segmentIndex) => {
                if (
                    applyFilters({
                        segmentIndex,
                        filtersData,
                        excludeFilterType: this.type,
                    })
                ) {
                    filteredSegments.push(segment);
                }

                return filteredSegments;
            },
            [],
        );
    }

    /**
     * Возвращает список доступных значений данного фильтра.
     *
     * @param segments
     */
    getOptions(segments: ITrainsTariffApiSegment[]): Option[] {
        return segments.reduce(
            (prevOptions, segment) => this.updateOptions(prevOptions, segment),
            this.getDefaultOptions(),
        );
    }

    /**
     * Возвращает список доступных значений для данного фильтра,
     * с учетом результатов других фильтров.
     */
    getActiveOptions({
        filtersData,
        segments,
        filteredSegments = this.getFilteredSegments({filtersData, segments}),
    }: {
        filtersData: ITrainsFilters;
        segments: ITrainsTariffApiSegment[];
        filteredSegments?: ITrainsTariffApiSegment[];
    }): Option[] {
        return filteredSegments.reduce(
            (options, segment) => this.updateOptions(options, segment),
            this.getDefaultOptions(),
        );
    }

    isAvailableWithOptions(
        options: Option[],
        _: ITrainsTariffApiSegment[] = [],
    ): boolean {
        return options.length > 1;
    }

    /**
     * Проверяет доступность данного фильтра для переданных доступных значений.
     */
    isAvailableWithActiveOptions(
        options: Option[],
        segments: ITrainsTariffApiSegment[] = [],
    ): boolean {
        return this.isAvailableWithOptions(options, segments);
    }

    /**
     * Возвращает список опций из данных обо всех фильтрах (из state).
     */
    getOptionsFromFiltersData(_filtersData: ITrainsFilters): Option[] {
        return [];
    }

    /**
     * Возвращает обновленное стостояние доступных значений фильтров.
     */
    updateActiveOptions({
        segments,
        filtersData,
    }: {
        segments: ITrainsTariffApiSegment[];
        filtersData: ITrainsFilters;
    }): ITrainsFilters {
        const filteredSegments = this.getFilteredSegments({
            segments,
            filtersData,
        });
        const activeOptions = this.getActiveOptions({
            filtersData,
            segments,
            filteredSegments,
        });
        const options = this.updateOptionsForFilteredSegments({
            options: this.getOptionsFromFiltersData(filtersData),
            filtersData,
            segments,
            filteredSegments,
        });

        return {
            ...filtersData,
            [this.type]: {
                ...filtersData[this.type],
                options,
                activeOptions,
                availableWithActiveOptions:
                    this.isAvailableWithActiveOptions(activeOptions),
            },
        };
    }

    /**
     * Возвращает массив, в котором ключом является индекс сегмента, а значение булевый признак,
     * того, что сегмент прошел фильтрацию по данному признаку.
     *
     * @param value - значение фильтра
     * @param segments - сегменты
     */
    getFilteredSegmentIndices(
        value: Value[],
        segments: ITrainsTariffApiSegment[],
    ): boolean[] {
        if (!this.shouldApply(value)) {
            return new Array(segments.length).fill(true);
        }

        return segments.map(segment => this.apply(value, segment));
    }

    reformatOptions(formattedOptions: Value[], _options: Option[]): Value[] {
        return formattedOptions;
    }

    validateValue(value: Value[], _options: Option[]): Value[] {
        return value;
    }

    setFilterValue({
        filtersData,
        value,
        segments,
    }: {
        filtersData: ITrainsFilters;
        value: Value[];
        segments: ITrainsTariffApiSegment[];
    }): ITrainsFilters {
        const options = this.getOptionsFromFiltersData(filtersData);

        const reformattedValue = this.reformatOptions(value, options);
        const validatedValue = this.validateValue(reformattedValue, options);

        return {
            ...filtersData,
            [this.type]: {
                ...filtersData[this.type],
                value: validatedValue,
                filteredSegmentIndices: this.getFilteredSegmentIndices(
                    validatedValue,
                    segments,
                ),
            },
        };
    }

    /**
     * Дополнительно преобразовывает текст опции для отображения с помощью функции.
     *
     * @param options
     * @param formatTextFn - функция для преобразования.
     */
    formatTextWithFn(
        options: {value: string; text: string}[],
        formatTextFn: (
            option: {value: string; text: string},
            index: number,
        ) => string,
    ): {value: string; text: string}[] {
        return options.map((option, index) => ({
            ...option,
            text: formatTextFn(option, index),
        }));
    }

    formatOptions(
        _options: Option[],
        _formatTextFn?: Function,
    ): {value: string; text: string}[] {
        return [];
    }

    /**
     * Производит преобразование ативных опций в массив с примитивными значениями для
     * последующей предачи в отображение.
     */
    formatActiveOptions(activeOptions: Option[]): string[] {
        return this.formatOptions(activeOptions).map(option => option.value);
    }
}
