import {
    ETrainsFilterType,
    ITrainsFilterOptionMinPrice,
    ITrainsFilterActiveOptionsAndOptionMinPriceList,
} from 'types/trains/search/filters/ITrainsFilters';
import {TrainsSearchContextType} from 'reducers/trains/context/types';
import {
    ITrainsVariant,
    ITrainsVariantAndDirection,
} from 'types/trains/common/variant/ITrainsVariant';
import {EQueryingStatus} from 'types/trains/search/searchInfo/ITrainsSearchInfo';
import {ITrainsSearchQueryParams} from 'types/trains/search/query/ITrainsSearchQueryParams';

import {getTrainsVariantsMinPrice} from 'projects/trains/lib/genericSearch/variants/getTrainsVariantsMinPrice';

class BaseFilterManager<Filter, Value, Option, Type extends ETrainsFilterType> {
    type: Type;
    initialFilter: Filter;

    constructor(type: Type, initialFilter: Filter) {
        this.type = type;
        this.initialFilter = initialFilter;
    }

    getDefaultValue(): Value[] {
        return [];
    }

    getDefaultOptions(): Option[] {
        return [];
    }

    checkIsDefaultValue(value: Value[]): boolean {
        return value.length === 0;
    }

    checkActiveValue(value: Value[]): boolean {
        return value.length > 0;
    }

    checkAvailableWithOptions(options: Option[]): boolean {
        return options.length > 1;
    }

    deserializeFromQuery(query: ITrainsSearchQueryParams): Value[] {
        if (!query) {
            return [];
        }

        return [];
    }

    getFilterByVariantsAndQuery({
        query,
        variants,
        context,
    }: {
        status: EQueryingStatus;
        variants: ITrainsVariant[];
        query: ITrainsSearchQueryParams;
        context: TrainsSearchContextType;
    }): Filter {
        const optionsByVariants = this.calculateOptionsByVariants({
            variants,
            context,
        });
        const value = this.getValueByQueryAndOptions({
            query,
            options: optionsByVariants,
        });

        return {
            ...this.initialFilter,
            value,
            options: optionsByVariants,
            activeOptions: optionsByVariants,
            availableWithOptions:
                this.checkAvailableWithOptions(optionsByVariants),
        };
    }

    calculateOptionsByVariants({
        variants,
    }: {
        variants: ITrainsVariant[];
        context: TrainsSearchContextType;
    }): Option[] {
        if (!variants) {
            return this.getDefaultOptions();
        }

        return this.getDefaultOptions();
    }

    getValueByQueryAndOptions({
        query,
        options,
    }: {
        query: ITrainsSearchQueryParams;
        options: Option[];
    }): Value[] {
        const valueFromQuery = this.deserializeFromQuery(query);
        const availableWithOptions = this.checkAvailableWithOptions(options);

        if (!availableWithOptions || !valueFromQuery) {
            return this.getDefaultValue();
        }

        return this.prepareValueFromQuery({valueFromQuery, options});
    }

    prepareValueFromQuery({
        valueFromQuery,
    }: {
        valueFromQuery: Value[];
        options: Option[];
    }): Value[] {
        return valueFromQuery;
    }

    prepareValueBeforeSaveToState({
        value,
    }: {
        filter: Filter;
        value: Value[];
    }): Value[] {
        return value;
    }

    calculateActiveOptionsAndMinPrices({
        options,
        variants,
        context,
    }: {
        options: Option[];
        variants: ITrainsVariant[];
        context: TrainsSearchContextType;
    }): ITrainsFilterActiveOptionsAndOptionMinPriceList<Option[]> {
        const {direction} = context;
        const activeOptions: Option[] = [];
        const optionMinPriceList: ITrainsFilterOptionMinPrice[] = [];
        let activeOptionMinPriceValue = Infinity;
        let totalOptionWithMinPriceValue = 0;

        options.forEach(option => {
            const filteredVariantsAndTariffs = variants.reduce<
                ITrainsVariant[]
            >((resultFilteredVariants, variant) => {
                const filterValue = this.getFilterValueByOption(option);
                const filteredVariant = this.filterVariantAndTariffs({
                    context,
                    value: filterValue,
                    variantAndDirection: {
                        variant,
                        direction,
                    },
                });

                if (filteredVariant) {
                    resultFilteredVariants.push(filteredVariant);
                }

                return resultFilteredVariants;
            }, []);

            if (!filteredVariantsAndTariffs.length) {
                optionMinPriceList.push({isMinPrice: false, price: undefined});

                return;
            }

            const minPrice = getTrainsVariantsMinPrice({
                variants: filteredVariantsAndTariffs,
                direction,
            });

            if (minPrice) {
                totalOptionWithMinPriceValue++;
            }

            const minPriceValue = minPrice?.value || Infinity;

            activeOptions.push(option);
            optionMinPriceList.push({isMinPrice: false, price: minPrice});
            activeOptionMinPriceValue = Math.min(
                activeOptionMinPriceValue,
                minPriceValue,
            );
        });

        return {
            activeOptions,
            optionMinPriceList:
                totalOptionWithMinPriceValue <= 1
                    ? optionMinPriceList
                    : optionMinPriceList.map(optionMinPrice => ({
                          price: optionMinPrice.price,
                          isMinPrice:
                              optionMinPrice.price?.value ===
                              activeOptionMinPriceValue,
                      })),
        };
    }

    getFilterValueByOption(option: Option): Value[] {
        if (!option) {
            return this.getDefaultValue();
        }

        return this.getDefaultValue();
    }

    filterVariantAndTariffs({
        variantAndDirection,
    }: {
        value: Value[];
        context: TrainsSearchContextType;
        variantAndDirection: ITrainsVariantAndDirection;
    }): ITrainsVariant | null {
        return variantAndDirection.variant;
    }
}

export default BaseFilterManager;
