import {
    ITrainsVariant,
    TTrainsVariantId,
    TTrainsVariantIdsByFilterType,
    ITrainsFilteredVariantsInfo,
    TTrainsVariantWithVisibleStatusById,
} from 'types/trains/common/variant/ITrainsVariant';
import {
    ETrainsFilterType,
    ITrainsFilters,
} from 'types/trains/search/filters/ITrainsFilters';
import {TrainsSearchContextType} from 'reducers/trains/context/types';

import {TTrainsFilterManagerTypes} from 'projects/trains/lib/genericSearch/filters/managers';

import {getFilterManagerByType} from './managers/getFilterManagerByType';
import {getActiveFilterManagerTypes} from './managers/getActiveFilterManagerTypes';

export type TFilterTypesByVariantId = Record<
    TTrainsVariantId,
    ETrainsFilterType[]
>;

/*
 * Готовим список id всех видимых вариантов.
 * */

const getVisibleVariantIds = (
    variantWithVisibleStatusById: TTrainsVariantWithVisibleStatusById,
): TTrainsVariantId[] => {
    return Object.values(variantWithVisibleStatusById)
        .filter(({isVisible}) => isVisible)
        .map(({variant}) => variant.id);
};

/*
 * Готовим для каждого типа фильтра пустую стуктуру для видимых id вариантов.
 * */

const fillEmptyFilterTypesWithVariantId = (
    variants: ITrainsVariant[],
): TFilterTypesByVariantId => {
    return variants.reduce<TFilterTypesByVariantId>(
        (filterTypesByVariantId, variant) => {
            filterTypesByVariantId[variant.id] = [];

            return filterTypesByVariantId;
        },
        {},
    );
};

/*
 * Для каждого типа фильтра получаем список id вариантов, которые будут отображаться, исключая данный фильтр.
 * */

const getVisibleVariantIdsWithFilterType = ({
    filterTypesByVariantId,
    activeFilterManagerTypes,
    variantWithVisibleStatusById,
}: {
    filterTypesByVariantId: TFilterTypesByVariantId;
    activeFilterManagerTypes: TTrainsFilterManagerTypes;
    variantWithVisibleStatusById: TTrainsVariantWithVisibleStatusById;
}): TTrainsVariantIdsByFilterType => {
    const visibleVariantIds = getVisibleVariantIds(
        variantWithVisibleStatusById,
    );

    return activeFilterManagerTypes.reduce<TTrainsVariantIdsByFilterType>(
        (variantIdsByFilterType, filterType) => {
            const filteredVariantIdsByFilter = Object.keys(
                filterTypesByVariantId,
            ).filter(variantId => {
                const filterTypes = filterTypesByVariantId[variantId];

                return (
                    filterTypes.length === 1 && filterTypes[0] === filterType
                );
            });

            variantIdsByFilterType[filterType] = [
                ...visibleVariantIds,
                ...filteredVariantIdsByFilter,
            ];

            return variantIdsByFilterType;
        },
        {},
    );
};

/*
 * Определяем, нужно ли отображать варианты и фильтруем тарифы в них.
 * Для каждого варианта получаем список типов фильтров, которые его отфильтровали.
 * */

export const getTrainsFilteredVariantsWithFilterTypesByManagerTypes = ({
    context,
    variants,
    filters,
    filterManagerTypes,
}: {
    filters: ITrainsFilters;
    variants: ITrainsVariant[];
    context: TrainsSearchContextType;
    filterManagerTypes: TTrainsFilterManagerTypes;
}): {
    filterTypesByVariantId: TFilterTypesByVariantId;
    variantWithVisibleStatusById: TTrainsVariantWithVisibleStatusById;
} => {
    const filterTypesByVariantId = fillEmptyFilterTypesWithVariantId(variants);
    const variantWithVisibleStatusById =
        variants.reduce<TTrainsVariantWithVisibleStatusById>(
            (variantById, variant) => {
                let variantWithFilteredTariffs = variant;
                let canVisibleCurrentVariant = true;

                filterManagerTypes.forEach(filterType => {
                    const filterManager = getFilterManagerByType(filterType);
                    const filter = filters[filterType];
                    const variantWithFilteredTariffsByFilter =
                        filterManager.filterVariantAndTariffs({
                            context,
                            value: filter.value as any,
                            variantAndDirection: {
                                variant: variantWithFilteredTariffs,
                                direction: context.direction,
                            },
                        });

                    if (variantWithFilteredTariffsByFilter) {
                        variantWithFilteredTariffs =
                            variantWithFilteredTariffsByFilter;
                    } else {
                        if (filterTypesByVariantId?.[variant.id]) {
                            filterTypesByVariantId[variant.id].push(filterType);
                        }

                        canVisibleCurrentVariant = false;
                    }
                }, variant);

                variantById[variant.id] = {
                    isVisible: canVisibleCurrentVariant,
                    variant: variantWithFilteredTariffs,
                };

                return variantById;
            },
            {},
        );

    return {
        filterTypesByVariantId,
        variantWithVisibleStatusById,
    };
};

/*
 * Определяем, нужно ли отображать варианты и фильтруем тарифы в них.
 * Для каждого фильтра получаем список видимых вариантов для данного фильтра, исключая при фильтрации текущий фильтр.
 * */

export const getTrainsFilteredVariantsInfo = ({
    context,
    filters,
    variants,
    originalVariantWithVisibleStatusById,
}: {
    filters: ITrainsFilters | null;
    variants: ITrainsVariant[];
    context: TrainsSearchContextType;
    originalVariantWithVisibleStatusById: TTrainsVariantWithVisibleStatusById;
}): ITrainsFilteredVariantsInfo => {
    const originalFilteredVariantsInfo: ITrainsFilteredVariantsInfo = {
        visibleVariantIdsByFilterType: null,
        variantWithVisibleStatusById: originalVariantWithVisibleStatusById,
    };

    if (!filters) {
        return originalFilteredVariantsInfo;
    }

    const activeFilterManagerTypes = getActiveFilterManagerTypes({filters});

    if (!activeFilterManagerTypes.length) {
        return originalFilteredVariantsInfo;
    }

    const {filterTypesByVariantId, variantWithVisibleStatusById} =
        getTrainsFilteredVariantsWithFilterTypesByManagerTypes({
            filters,
            context,
            variants,
            filterManagerTypes: activeFilterManagerTypes,
        });

    const visibleVariantIdsByFilterType = getVisibleVariantIdsWithFilterType({
        filterTypesByVariantId,
        activeFilterManagerTypes,
        variantWithVisibleStatusById,
    });

    return {
        variantWithVisibleStatusById,
        visibleVariantIdsByFilterType,
    };
};
