import _intersection from 'lodash/intersection';

import {TIMES_OF_DAY} from 'utilities/dateUtils/constants';

import {
    ETrainsFilterType,
    ITrainsArrivalFilter,
} from 'types/trains/search/filters/ITrainsFilters';
import {ETimeOfDay} from 'utilities/dateUtils/types';
import {TrainsSearchContextType} from 'reducers/trains/context/types';
import {ITrainsSearchQueryParams} from 'types/trains/search/query/ITrainsSearchQueryParams';
import {
    ITrainsVariant,
    ITrainsVariantAndDirection,
} from 'types/trains/common/variant/ITrainsVariant';

import {getTimeOfDay} from 'utilities/dateUtils';
import {isTimeOfDay} from 'utilities/dateUtils/isTimeOfDay';
import {getTrainsVariantLocalArrivalTime} from 'projects/trains/lib/genericSearch/variants/getTrainsVariantArrival';

import BaseFilterManager from '../../baseFilterManager/BaseFilterManager';

const initialArrivalFilter: ITrainsArrivalFilter = {
    value: [],
    options: TIMES_OF_DAY,
    activeOptions: [],
    availableWithOptions: false,
    availableWithActiveOptions: false,
    type: ETrainsFilterType.ARRIVAL,
    filteredSegmentIndices: [],
};

class ArrivalFilterManager extends BaseFilterManager<
    ITrainsArrivalFilter,
    ETimeOfDay,
    ETimeOfDay,
    ETrainsFilterType.ARRIVAL
> {
    constructor() {
        super(ETrainsFilterType.ARRIVAL, initialArrivalFilter);
    }

    /* Fill filter */

    prepareValueFromQuery({
        valueFromQuery,
        options,
    }: {
        valueFromQuery: ETimeOfDay[];
        options: ETimeOfDay[];
    }): ETimeOfDay[] {
        return _intersection(valueFromQuery, options);
    }

    getDefaultOptions(): ETimeOfDay[] {
        return TIMES_OF_DAY;
    }

    getFilterValueByOption(option: ETimeOfDay): ETimeOfDay[] {
        return [option];
    }

    /* Apply filter */

    filterVariantAndTariffs({
        value,
        variantAndDirection,
    }: {
        value: ETimeOfDay[];
        context: TrainsSearchContextType;
        variantAndDirection: ITrainsVariantAndDirection;
    }): ITrainsVariant | null {
        if (!value.length) {
            return variantAndDirection.variant;
        }

        const variantLocalArrivalTime =
            getTrainsVariantLocalArrivalTime(variantAndDirection);

        if (!variantLocalArrivalTime) {
            return null;
        }

        const variantLocalArrivalHour = variantLocalArrivalTime.hour();
        const arrivalTimeOfDay = getTimeOfDay(variantLocalArrivalHour);
        const canVisibleVariant =
            arrivalTimeOfDay && value.includes(arrivalTimeOfDay);

        if (!canVisibleVariant) {
            return null;
        }

        return variantAndDirection.variant;
    }

    /* Sync filter with query */

    deserializeFromQuery({arrival}: ITrainsSearchQueryParams): ETimeOfDay[] {
        if (!arrival) {
            return [];
        }

        const valueOptions: string[] = Array.isArray(arrival)
            ? [...new Set(arrival)]
            : [arrival];

        return valueOptions.filter(isTimeOfDay);
    }

    serializeToQuery(arrival: ETimeOfDay[]): {arrival?: ETimeOfDay[]} {
        if (this.checkIsDefaultValue(arrival)) {
            return {};
        }

        return {arrival};
    }
}

export default ArrivalFilterManager;
