import _intersection from 'lodash/intersection';

import {TIMES_OF_DAY} from 'utilities/dateUtils/constants';

import {
    ETrainsFilterType,
    ITrainsDepartureFilter,
} from 'types/trains/search/filters/ITrainsFilters';
import {ETimeOfDay} from 'utilities/dateUtils/types';
import {TrainsSearchContextType} from 'reducers/trains/context/types';
import {ITrainsSearchQueryParams} from 'types/trains/search/query/ITrainsSearchQueryParams';
import {
    ITrainsVariant,
    ITrainsVariantAndDirection,
} from 'types/trains/common/variant/ITrainsVariant';

import {isTimeOfDay} from 'utilities/dateUtils/isTimeOfDay';
import {getTrainsVariantLocalDepartureTimeOfDay} from 'projects/trains/lib/genericSearch/variants/getTrainsVariantDepartureTimeOfDay';

import BaseFilterManager from '../../baseFilterManager/BaseFilterManager';

const initialDepartureFilter: ITrainsDepartureFilter = {
    value: [],
    options: TIMES_OF_DAY,
    activeOptions: [],
    availableWithOptions: false,
    availableWithActiveOptions: false,
    type: ETrainsFilterType.DEPARTURE,
    filteredSegmentIndices: [],
};

class DepartureFilterManager extends BaseFilterManager<
    ITrainsDepartureFilter,
    ETimeOfDay,
    ETimeOfDay,
    ETrainsFilterType.DEPARTURE
> {
    constructor() {
        super(ETrainsFilterType.DEPARTURE, initialDepartureFilter);
    }

    /* Fill filter */

    prepareValueFromQuery({
        valueFromQuery,
        options,
    }: {
        valueFromQuery: ETimeOfDay[];
        options: ETimeOfDay[];
    }): ETimeOfDay[] {
        return _intersection(valueFromQuery, options);
    }

    getDefaultOptions(): ETimeOfDay[] {
        return TIMES_OF_DAY;
    }

    getFilterValueByOption(option: ETimeOfDay): ETimeOfDay[] {
        return [option];
    }

    /* Apply filter */

    filterVariantAndTariffs({
        value,
        context,
        variantAndDirection,
    }: {
        value: ETimeOfDay[];
        context: TrainsSearchContextType;
        variantAndDirection: ITrainsVariantAndDirection;
    }): ITrainsVariant | null {
        if (!value.length) {
            return variantAndDirection.variant;
        }

        const variantLocalDepartureTimeOfDay =
            getTrainsVariantLocalDepartureTimeOfDay({
                context,
                variantAndDirection,
            });

        if (!variantLocalDepartureTimeOfDay) {
            return null;
        }

        const canVisibleVariant = value.includes(
            variantLocalDepartureTimeOfDay,
        );

        if (!canVisibleVariant) {
            return null;
        }

        return variantAndDirection.variant;
    }

    /* Sync filter with query */

    deserializeFromQuery({departure}: ITrainsSearchQueryParams): ETimeOfDay[] {
        if (!departure) {
            return [];
        }

        const valueOptions: string[] = Array.isArray(departure)
            ? [...new Set(departure)]
            : [departure];

        return valueOptions.filter(isTimeOfDay);
    }

    serializeToQuery(departure: ETimeOfDay[]): {departure?: ETimeOfDay[]} {
        if (this.checkIsDefaultValue(departure)) {
            return {};
        }

        return {departure};
    }
}

export default DepartureFilterManager;
