import {
    ETrainsFilterType,
    ITrainsHideWithoutPriceFilter,
    ITrainsHideWithoutPriceOptions,
    ITrainsFilterActiveOptionsAndOptionMinPriceList,
} from 'types/trains/search/filters/ITrainsFilters';
import {TrainsSearchContextType} from 'reducers/trains/context/types';
import {EQueryingStatus} from 'types/trains/search/searchInfo/ITrainsSearchInfo';
import {ITrainsSearchQueryParams} from 'types/trains/search/query/ITrainsSearchQueryParams';
import {
    ITrainsVariant,
    ITrainsVariantAndDirection,
} from 'types/trains/common/variant/ITrainsVariant';

import {checkHasTrainsVariantTariffClasses} from 'projects/trains/lib/genericSearch/variants/checkHasTrainsVariantTariffClasses';
import {checkTrainsVariantsWithAndWithoutPrice} from 'projects/trains/lib/genericSearch/variants/checkTrainsVariantsWithAndWithoutPrice';

const initialHideWithoutPriceOptions: ITrainsHideWithoutPriceOptions = {
    withoutPrice: false,
    withPrice: false,
};

const initialHideWithoutPriceFilterManager: ITrainsHideWithoutPriceFilter = {
    value: false,
    options: initialHideWithoutPriceOptions,
    activeOptions: initialHideWithoutPriceOptions,
    availableWithActiveOptions: false,
    availableWithOptions: false,
    type: ETrainsFilterType.HIDE_WITHOUT_PRICE,
    filteredSegmentIndices: [],
};

const SEATS_QUERY_VALUE = 'y';

class HideWithoutPriceFilterManager {
    type: ETrainsFilterType.HIDE_WITHOUT_PRICE =
        ETrainsFilterType.HIDE_WITHOUT_PRICE;

    /* Fill filter */

    getFilterByVariantsAndQuery({
        query,
        status,
        variants,
        context,
    }: {
        status: EQueryingStatus;
        variants: ITrainsVariant[];
        query: ITrainsSearchQueryParams;
        context: TrainsSearchContextType;
    }): ITrainsHideWithoutPriceFilter {
        const optionsByVariants =
            status === EQueryingStatus.DONE
                ? this.calculateOptionsByVariants({variants, context})
                : initialHideWithoutPriceOptions;

        const value = this.getValueByQueryAndOptions({
            query,
            options: optionsByVariants,
        });

        return {
            ...initialHideWithoutPriceFilterManager,
            value,
            options: optionsByVariants,
            activeOptions: optionsByVariants,
        };
    }

    getValueByQueryAndOptions({
        query,
        options,
    }: {
        query: ITrainsSearchQueryParams;
        options: ITrainsHideWithoutPriceOptions;
    }): boolean {
        const valueFromQuery = this.deserializeFromQuery(query);
        const availableWithOptions = this.checkAvailableWithOptions(options);

        if (!availableWithOptions || !valueFromQuery) {
            return this.getDefaultValue();
        }

        return this.prepareValueFromQuery({valueFromQuery, options});
    }

    calculateActiveOptionsAndMinPrices({
        variants,
        context,
    }: {
        options: ITrainsHideWithoutPriceOptions;
        variants: ITrainsVariant[];
        context: TrainsSearchContextType;
    }): ITrainsFilterActiveOptionsAndOptionMinPriceList<ITrainsHideWithoutPriceOptions> {
        const {direction} = context;
        const {hasVariantWithoutPrice, hasVariantWithPrice} =
            checkTrainsVariantsWithAndWithoutPrice({
                variants,
                direction,
            });

        if (hasVariantWithPrice && hasVariantWithoutPrice) {
            return {
                activeOptions: {
                    withPrice: true,
                    withoutPrice: true,
                },
                optionMinPriceList: [],
            };
        }

        return {
            activeOptions: initialHideWithoutPriceOptions,
            optionMinPriceList: [],
        };
    }

    prepareValueFromQuery({
        valueFromQuery,
    }: {
        valueFromQuery: boolean;
        options: ITrainsHideWithoutPriceOptions;
    }): boolean {
        return valueFromQuery;
    }

    checkAvailableWithOptions(
        options: ITrainsHideWithoutPriceOptions,
    ): boolean {
        return options.withPrice && options.withoutPrice;
    }

    calculateOptionsByVariants({
        variants,
        context,
    }: {
        variants: ITrainsVariant[];
        context: TrainsSearchContextType;
    }): ITrainsHideWithoutPriceOptions {
        const {direction} = context;
        const {hasVariantWithoutPrice, hasVariantWithPrice} =
            checkTrainsVariantsWithAndWithoutPrice({
                variants,
                direction,
            });

        if (hasVariantWithPrice && hasVariantWithoutPrice) {
            return {
                withPrice: true,
                withoutPrice: true,
            };
        }

        return initialHideWithoutPriceOptions;
    }

    getDefaultValue(): boolean {
        return false;
    }

    checkIsDefaultValue(value: boolean): boolean {
        return value === this.getDefaultValue();
    }

    prepareValueBeforeSaveToState({
        value,
    }: {
        filter: ITrainsHideWithoutPriceFilter;
        value: boolean;
    }): boolean {
        return value;
    }

    /* Apply filter */

    filterVariantAndTariffs({
        value,
        variantAndDirection,
    }: {
        value: boolean;
        context: TrainsSearchContextType;
        variantAndDirection: ITrainsVariantAndDirection;
    }): ITrainsVariant | null {
        if (!value) {
            return variantAndDirection.variant;
        }

        const canVisibleVariant =
            checkHasTrainsVariantTariffClasses(variantAndDirection);

        if (!canVisibleVariant) {
            return null;
        }

        return variantAndDirection.variant;
    }

    checkActiveValue(value: boolean): boolean {
        return value !== this.getDefaultValue();
    }

    /* Sync filter with query */

    deserializeFromQuery({seats}: ITrainsSearchQueryParams): boolean {
        return (
            typeof seats === 'string' &&
            seats.toLowerCase() === SEATS_QUERY_VALUE
        );
    }

    serializeToQuery(withPriceOnly: boolean): {
        seats?: ITrainsSearchQueryParams['seats'];
    } {
        if (this.checkIsDefaultValue(withPriceOnly)) {
            return {};
        }

        return {
            seats: SEATS_QUERY_VALUE,
        };
    }
}

export default HideWithoutPriceFilterManager;
