import _intersection from 'lodash/intersection';

import {
    ETrainsFilterType,
    ITrainsHighSpeedTrainFilter,
    ITrainsSimpleFilterOption,
} from 'types/trains/search/filters/ITrainsFilters';
import {TrainsSearchContextType} from 'reducers/trains/context/types';
import {ITrainsSearchQueryParams} from 'types/trains/search/query/ITrainsSearchQueryParams';
import {
    ITrainsVariant,
    ITrainsVariantAndDirection,
} from 'types/trains/common/variant/ITrainsVariant';

import BaseFilterManager from 'projects/trains/lib/genericSearch/filters/baseFilterManager/BaseFilterManager';
import {getTrainsVariantHighSpeedTrains} from 'projects/trains/lib/genericSearch/variants/getTrainsVariantHighSpeedTrains';

const initialHighSpeedTrainFilter: ITrainsHighSpeedTrainFilter = {
    value: [],
    options: [],
    activeOptions: [],
    availableWithOptions: false,
    availableWithActiveOptions: false,
    type: ETrainsFilterType.HIGH_SPEED_TRAIN,
    filteredSegmentIndices: [],
};

class HighSpeedTrainFilterManager extends BaseFilterManager<
    ITrainsHighSpeedTrainFilter,
    string,
    ITrainsSimpleFilterOption,
    ETrainsFilterType.HIGH_SPEED_TRAIN
> {
    constructor() {
        super(ETrainsFilterType.HIGH_SPEED_TRAIN, initialHighSpeedTrainFilter);
    }

    /* Fill filter */

    prepareValueFromQuery({
        valueFromQuery,
        options,
    }: {
        valueFromQuery: string[];
        options: ITrainsSimpleFilterOption[];
    }): string[] {
        const optionValues = options.map(({value}) => value);

        return _intersection(valueFromQuery, optionValues);
    }

    calculateOptionsByVariants({
        variants,
        context,
    }: {
        variants: ITrainsVariant[];
        context: TrainsSearchContextType;
    }): ITrainsSimpleFilterOption[] {
        const {direction} = context;
        const highSpeedTrainIdsSet = new Set<string>();

        return variants.reduce<ITrainsSimpleFilterOption[]>(
            (highSpeedTrainOptions, variant) => {
                const variantHighSpeedTrains = getTrainsVariantHighSpeedTrains({
                    variant,
                    direction,
                });

                variantHighSpeedTrains.forEach(variantHighSpeedTrain => {
                    if (variantHighSpeedTrain) {
                        const {id, title} = variantHighSpeedTrain;

                        if (!highSpeedTrainIdsSet.has(id)) {
                            highSpeedTrainOptions.push({
                                value: id,
                                text: title,
                            });
                        }

                        highSpeedTrainIdsSet.add(id);
                    }
                });

                return highSpeedTrainOptions;
            },
            [],
        );
    }

    getFilterValueByOption(option: ITrainsSimpleFilterOption): string[] {
        return [option.value];
    }

    checkAvailableWithOptions(options: ITrainsSimpleFilterOption[]): boolean {
        return options.length > 0;
    }

    /* Apply filter */

    filterVariantAndTariffs({
        value,
        variantAndDirection,
    }: {
        value: string[];
        context: TrainsSearchContextType;
        variantAndDirection: ITrainsVariantAndDirection;
    }): ITrainsVariant | null {
        if (!value.length) {
            return variantAndDirection.variant;
        }

        const variantHighSpeedTrains =
            getTrainsVariantHighSpeedTrains(variantAndDirection);

        const canVisibleVariant = variantHighSpeedTrains.some(
            variantHighSpeedTrain => {
                if (!variantHighSpeedTrain) {
                    return false;
                }

                const {id} = variantHighSpeedTrain;

                return value.includes(id);
            },
        );

        if (!canVisibleVariant) {
            return null;
        }

        return variantAndDirection.variant;
    }

    /* Sync filter with query */

    deserializeFromQuery({highSpeedTrain}: ITrainsSearchQueryParams): string[] {
        if (!highSpeedTrain) {
            return this.getDefaultValue();
        }

        return Array.isArray(highSpeedTrain)
            ? [...new Set(highSpeedTrain)]
            : [highSpeedTrain];
    }

    serializeToQuery(value: string[]): {highSpeedTrain?: string[]} {
        if (this.checkIsDefaultValue(value)) {
            return {};
        }

        return {highSpeedTrain: value};
    }
}

export default HighSpeedTrainFilterManager;
