import {
    ETrainsFilterType,
    ITrainsPriceRangeFilter,
    ITrainsPriceRangeOption,
} from 'types/trains/search/filters/ITrainsFilters';
import {isNotNull} from 'types/utilities';
import {TrainsSearchContextType} from 'reducers/trains/context/types';
import {ITrainsSearchQueryParams} from 'types/trains/search/query/ITrainsSearchQueryParams';
import {
    ITrainsVariant,
    ITrainsVariantAndDirection,
} from 'types/trains/common/variant/ITrainsVariant';

import {CurrencyType} from 'utilities/currency/CurrencyType';
import {ceilToFrequencyRate} from './utilities/ceilToFrequencyRate';
import {floorToFrequencyRate} from './utilities/floorToFrequencyRate';
import {getTrainsVariantsMinPrice} from 'projects/trains/lib/genericSearch/variants/getTrainsVariantsMinPrice';
import {getTrainsVariantsMaxPrice} from 'projects/trains/lib/genericSearch/variants/getTrainsVariantsMaxPrice';
import {checkAndFilterVariantTariffsByPriceRange} from 'projects/trains/lib/genericSearch/tariffs/checkAndFilterVariantTariffsByPriceRange';

import BaseFilterManager from '../../baseFilterManager/BaseFilterManager';

export interface ITrainsPriceRange {
    min: number;
    max: number;
}

const initialPriceRangeFilter: ITrainsPriceRangeFilter = {
    value: [],
    options: [],
    activeOptions: [],
    availableWithOptions: false,
    availableWithActiveOptions: false,
    type: ETrainsFilterType.PRICE_RANGE,
    filteredSegmentIndices: [],
};

const FREQUENCY_RATE = 100;
const PRICE_RANGE_SEPARATOR = '-';

class PriceRangeFilterManager extends BaseFilterManager<
    ITrainsPriceRangeFilter,
    ITrainsPriceRangeOption,
    ITrainsPriceRangeOption,
    ETrainsFilterType.PRICE_RANGE
> {
    constructor() {
        super(ETrainsFilterType.PRICE_RANGE, initialPriceRangeFilter);
    }

    /* Fill filter */

    getRangeValueByMinAndMax = (priceRange: ITrainsPriceRange): string => {
        const {min, max} = priceRange;

        return `${min}${PRICE_RANGE_SEPARATOR}${max}`;
    };

    prepareValueFromQuery({
        valueFromQuery,
        options,
    }: {
        valueFromQuery: ITrainsPriceRangeOption[];
        options: ITrainsPriceRangeOption[];
    }): ITrainsPriceRangeOption[] {
        const [firstOption] = options;
        const [firstValue] = valueFromQuery;

        if (!firstOption || !firstValue) {
            return this.getDefaultValue();
        }

        if (
            firstValue.min >= firstOption.min &&
            firstValue.max <= firstOption.max
        ) {
            const {min, max} = firstValue;

            return [
                {
                    min,
                    max,
                    count: 0,
                    currency: firstOption.currency,
                    value: this.getRangeValueByMinAndMax({min, max}),
                },
            ];
        }

        return this.getDefaultValue();
    }

    checkAvailableWithOptions(
        optionsByVariants: ITrainsPriceRangeOption[],
    ): boolean {
        return optionsByVariants.length > 0;
    }

    calculateOptionsByVariants({
        variants,
        context,
    }: {
        variants: ITrainsVariant[];
        context: TrainsSearchContextType;
    }): ITrainsPriceRangeOption[] {
        const {direction} = context;
        const minPrice = getTrainsVariantsMinPrice({
            variants,
            direction,
        });
        const maxPrice = getTrainsVariantsMaxPrice({
            variants,
            direction,
        });

        if (!minPrice?.value || !maxPrice?.value) {
            return [];
        }

        if (
            minPrice.value >= maxPrice.value ||
            minPrice.currency !== maxPrice.currency
        ) {
            return [];
        }

        const min = floorToFrequencyRate({
            value: minPrice.value,
            frequencyRate: FREQUENCY_RATE,
        });
        const max = ceilToFrequencyRate({
            value: maxPrice.value,
            frequencyRate: FREQUENCY_RATE,
        });

        return [
            {
                min,
                max,
                count: 0,
                currency: minPrice.currency,
                value: this.getRangeValueByMinAndMax({min, max}),
            },
        ];
    }

    prepareValueBeforeSaveToState({
        filter,
        value,
    }: {
        filter: ITrainsPriceRangeFilter;
        value: ITrainsPriceRangeOption[];
    }): ITrainsPriceRangeOption[] {
        const {options} = filter;

        if (!options.length || !value.length) {
            return this.getDefaultValue();
        }

        const [firstValue] = value;
        const [firstOption] = options;

        if (
            firstValue.min === firstOption.min &&
            firstValue.max === firstOption.max
        ) {
            return this.getDefaultValue();
        }

        const {min, max} = firstValue;

        return [
            {
                min,
                max,
                count: 0,
                value: this.getRangeValueByMinAndMax({min, max}),
                currency: firstValue.currency ?? CurrencyType.RUB,
            },
        ];
    }

    /* Apply filter */

    filterVariantAndTariffs({
        value,
        variantAndDirection,
    }: {
        value: ITrainsPriceRangeOption[];
        context: TrainsSearchContextType;
        variantAndDirection: ITrainsVariantAndDirection;
    }): ITrainsVariant | null {
        if (!value.length) {
            return variantAndDirection.variant;
        }

        const [priceRange] = value;

        return checkAndFilterVariantTariffsByPriceRange({
            priceRange,
            variantAndDirection,
        });
    }

    /* Sync filter with query */

    parseOptionFromString(value: string): ITrainsPriceRangeOption | null {
        const priceRange = value
            .split(PRICE_RANGE_SEPARATOR)
            .map(priceString => parseInt(priceString.trim(), 10))
            .filter(priceNumber => !isNaN(priceNumber));

        if (priceRange.length !== 2) {
            return null;
        }

        const [min, max] = priceRange;

        if (min < 0 || max < 0 || min > max) {
            return null;
        }

        return {
            min,
            max,
            count: 0,
            currency: CurrencyType.RUB,
            value: this.getRangeValueByMinAndMax({min, max}),
        };
    }

    deserializeFromQuery({
        priceRange,
    }: ITrainsSearchQueryParams): ITrainsPriceRangeOption[] {
        if (!priceRange) {
            return [];
        }

        const valueOptions = Array.isArray(priceRange)
            ? priceRange
            : [priceRange];

        return valueOptions
            .map(valueOption => valueOption.trim())
            .map(value => this.parseOptionFromString(value))
            .filter(isNotNull);
    }

    serializeToQuery(priceRangeOptions: ITrainsPriceRangeOption[]): {
        priceRange?: string[];
    } {
        if (this.checkIsDefaultValue(priceRangeOptions)) {
            return {};
        }

        const priceRange = priceRangeOptions.map(option => option.value);

        return {
            priceRange,
        };
    }
}

export default PriceRangeFilterManager;
