import _intersection from 'lodash/intersection';

import {
    ETrainsFilterType,
    ITrainsSimpleFilterOption,
    ITrainsStationFromFilter,
} from 'types/trains/search/filters/ITrainsFilters';
import {TrainsSearchContextType} from 'reducers/trains/context/types';
import {ITrainsSearchQueryParams} from 'types/trains/search/query/ITrainsSearchQueryParams';
import {
    ITrainsVariant,
    ITrainsVariantAndDirection,
} from 'types/trains/common/variant/ITrainsVariant';

import BaseFilterManager from 'projects/trains/lib/genericSearch/filters/baseFilterManager/BaseFilterManager';
import {getTrainsVariantFirstSegment} from 'projects/trains/lib/genericSearch/variants/getTrainsVariantFirstSegment';

const initialStationFromFilter: ITrainsStationFromFilter = {
    value: [],
    options: [],
    activeOptions: [],
    availableWithOptions: false,
    availableWithActiveOptions: false,
    type: ETrainsFilterType.STATION_FROM,
    filteredSegmentIndices: [],
};

class StationFromFilterManager extends BaseFilterManager<
    ITrainsStationFromFilter,
    string,
    ITrainsSimpleFilterOption,
    ETrainsFilterType.STATION_FROM
> {
    constructor() {
        super(ETrainsFilterType.STATION_FROM, initialStationFromFilter);
    }

    /* Fill filter */

    prepareValueFromQuery({
        valueFromQuery,
        options,
    }: {
        valueFromQuery: string[];
        options: ITrainsSimpleFilterOption[];
    }): string[] {
        const optionValues = options.map(({value}) => value);

        return _intersection(valueFromQuery, optionValues);
    }

    calculateOptionsByVariants({
        variants,
        context,
    }: {
        variants: ITrainsVariant[];
        context: TrainsSearchContextType;
    }): ITrainsSimpleFilterOption[] {
        const {direction} = context;
        const stationFromIdsSet = new Set<string>();

        return variants.reduce<ITrainsSimpleFilterOption[]>(
            (stationFromOptions, variant) => {
                const firstSegment = getTrainsVariantFirstSegment({
                    variant,
                    direction,
                });

                if (firstSegment) {
                    const {stationFrom} = firstSegment;
                    const {id, title} = stationFrom;

                    if (!stationFromIdsSet.has(String(id))) {
                        stationFromOptions.push({
                            value: String(id),
                            text: title,
                        });
                    }

                    stationFromIdsSet.add(String(id));
                }

                return stationFromOptions;
            },
            [],
        );
    }

    getFilterValueByOption(option: ITrainsSimpleFilterOption): string[] {
        return [option.value];
    }

    /* Apply filter */

    filterVariantAndTariffs({
        value,
        variantAndDirection,
    }: {
        value: string[];
        context: TrainsSearchContextType;
        variantAndDirection: ITrainsVariantAndDirection;
    }): ITrainsVariant | null {
        if (!value.length) {
            return variantAndDirection.variant;
        }

        const firstSegment = getTrainsVariantFirstSegment(variantAndDirection);

        if (!firstSegment) {
            return null;
        }

        const canVisibleVariant = value.some(
            stationId => stationId === String(firstSegment.stationFrom.id),
        );

        if (!canVisibleVariant) {
            return null;
        }

        return variantAndDirection.variant;
    }

    /* Sync filter with query */

    deserializeFromQuery({stationFrom}: ITrainsSearchQueryParams): string[] {
        if (!stationFrom) {
            return this.getDefaultValue();
        }

        return Array.isArray(stationFrom)
            ? [...new Set(stationFrom)]
            : [stationFrom];
    }

    serializeToQuery(stationFrom: string[]): {stationFrom?: string[]} {
        if (this.checkIsDefaultValue(stationFrom)) {
            return {};
        }

        return {stationFrom};
    }
}

export default StationFromFilterManager;
