import _intersection from 'lodash/intersection';

import {
    ETrainsFilterType,
    ITrainsStationToFilter,
    ITrainsSimpleFilterOption,
} from 'types/trains/search/filters/ITrainsFilters';
import {TrainsSearchContextType} from 'reducers/trains/context/types';
import {ITrainsSearchQueryParams} from 'types/trains/search/query/ITrainsSearchQueryParams';
import {
    ITrainsVariant,
    ITrainsVariantAndDirection,
} from 'types/trains/common/variant/ITrainsVariant';

import {getTrainsVariantLastSegment} from 'projects/trains/lib/genericSearch/variants/getTrainsVariantLastSegment';

import BaseFilterManager from '../../baseFilterManager/BaseFilterManager';

const initialStationToFilter: ITrainsStationToFilter = {
    value: [],
    options: [],
    activeOptions: [],
    availableWithOptions: false,
    availableWithActiveOptions: false,
    type: ETrainsFilterType.STATION_TO,
    filteredSegmentIndices: [],
};

class StationToFilterManager extends BaseFilterManager<
    ITrainsStationToFilter,
    string,
    ITrainsSimpleFilterOption,
    ETrainsFilterType.STATION_TO
> {
    constructor() {
        super(ETrainsFilterType.STATION_TO, initialStationToFilter);
    }

    /* Fill filter */

    prepareValueFromQuery({
        valueFromQuery,
        options,
    }: {
        valueFromQuery: string[];
        options: ITrainsSimpleFilterOption[];
    }): string[] {
        const optionValues = options.map(({value}) => value);

        return _intersection(valueFromQuery, optionValues);
    }

    calculateOptionsByVariants({
        variants,
        context,
    }: {
        variants: ITrainsVariant[];
        context: TrainsSearchContextType;
    }): ITrainsSimpleFilterOption[] {
        const {direction} = context;
        const stationToIdsSet = new Set<string>();

        return variants.reduce<ITrainsSimpleFilterOption[]>(
            (stationToOptions, variant) => {
                const lastSegment = getTrainsVariantLastSegment({
                    variant,
                    direction,
                });

                if (lastSegment) {
                    const {stationTo} = lastSegment;
                    const {id, title} = stationTo;

                    if (!stationToIdsSet.has(String(id))) {
                        stationToOptions.push({
                            value: String(id),
                            text: title,
                        });
                    }

                    stationToIdsSet.add(String(id));
                }

                return stationToOptions;
            },
            [],
        );
    }

    getFilterValueByOption(option: ITrainsSimpleFilterOption): string[] {
        return [option.value];
    }

    /* Apply filter */

    filterVariantAndTariffs({
        value,
        variantAndDirection,
    }: {
        value: string[];
        context: TrainsSearchContextType;
        variantAndDirection: ITrainsVariantAndDirection;
    }): ITrainsVariant | null {
        if (!value.length) {
            return variantAndDirection.variant;
        }

        const lastSegment = getTrainsVariantLastSegment(variantAndDirection);

        if (!lastSegment) {
            return null;
        }

        const canVisibleVariant = value.some(
            stationId => stationId === String(lastSegment?.stationTo?.id),
        );

        if (!canVisibleVariant) {
            return null;
        }

        return variantAndDirection.variant;
    }

    /* Sync filter with query */

    deserializeFromQuery({stationTo}: ITrainsSearchQueryParams): string[] {
        if (!stationTo) {
            return this.getDefaultValue();
        }

        return Array.isArray(stationTo) ? [...new Set(stationTo)] : [stationTo];
    }

    serializeToQuery(stationTo: string[]): {stationTo?: string[]} {
        if (this.checkIsDefaultValue(stationTo)) {
            return {};
        }

        return {stationTo};
    }
}

export default StationToFilterManager;
