import _intersection from 'lodash/intersection';

import {TRAIN_TARIFF_CLASSES} from 'projects/trains/lib/segments/tariffs/constants';

import {
    ETrainsFilterType,
    ITrainsTrainTariffClassFilter,
} from 'types/trains/search/filters/ITrainsFilters';
import {TrainsSearchContextType} from 'reducers/trains/context/types';
import {ETrainsCoachType} from 'types/trains/common/tariffs/ETrainsCoachType';
import {ITrainsSearchQueryParams} from 'types/trains/search/query/ITrainsSearchQueryParams';
import {
    ITrainsVariant,
    ITrainsVariantAndDirection,
} from 'types/trains/common/variant/ITrainsVariant';

import BaseFilterManager from 'projects/trains/lib/genericSearch/filters/baseFilterManager/BaseFilterManager';
import {getTrainsVariantsCoachTypes} from 'projects/trains/lib/genericSearch/variants/getTrainsVariantsCoachTypes';
import {checkAndFilterTrainsVariantTariffsByCoachTypes} from 'projects/trains/lib/genericSearch/tariffs/checkAndFilterVariantTariffsByTariffClasses';

const initialTrainTariffClassFilter: ITrainsTrainTariffClassFilter = {
    value: [],
    options: [],
    activeOptions: [],
    availableWithOptions: false,
    availableWithActiveOptions: false,
    type: ETrainsFilterType.TRAIN_TARIFF_CLASS,
    filteredSegmentIndices: [],
};

class TrainTariffClassFilterManager extends BaseFilterManager<
    ITrainsTrainTariffClassFilter,
    ETrainsCoachType,
    ETrainsCoachType,
    ETrainsFilterType.TRAIN_TARIFF_CLASS
> {
    constructor() {
        super(
            ETrainsFilterType.TRAIN_TARIFF_CLASS,
            initialTrainTariffClassFilter,
        );
    }

    /* Fill filter */

    prepareValueFromQuery({
        valueFromQuery,
        options,
    }: {
        valueFromQuery: ETrainsCoachType[];
        options: ETrainsCoachType[];
    }): ETrainsCoachType[] {
        return _intersection(valueFromQuery, options);
    }

    calculateOptionsByVariants({
        variants,
        context,
    }: {
        variants: ITrainsVariant[];
        context: TrainsSearchContextType;
    }): ETrainsCoachType[] {
        const {direction} = context;

        return getTrainsVariantsCoachTypes({
            variants,
            direction,
        });
    }

    getFilterValueByOption(option: ETrainsCoachType): ETrainsCoachType[] {
        return [option];
    }

    /* Apply filter */

    filterVariantAndTariffs({
        value,
        variantAndDirection,
    }: {
        value: ETrainsCoachType[];
        context: TrainsSearchContextType;
        variantAndDirection: ITrainsVariantAndDirection;
    }): ITrainsVariant | null {
        if (!value.length) {
            return variantAndDirection.variant;
        }

        return checkAndFilterTrainsVariantTariffsByCoachTypes({
            coachTypes: value,
            variantAndDirection,
        });
    }

    /* Sync filter with query */

    deserializeFromQuery({
        trainTariffClass,
    }: ITrainsSearchQueryParams): ETrainsCoachType[] {
        if (!trainTariffClass) {
            return this.getDefaultValue();
        }

        const queryClasses = Array.isArray(trainTariffClass)
            ? [...new Set(trainTariffClass)]
            : [trainTariffClass];

        return TRAIN_TARIFF_CLASSES.filter(tariffClass =>
            queryClasses.includes(tariffClass),
        );
    }

    serializeToQuery(trainTariffClass: ETrainsCoachType[]): {
        trainTariffClass?: ETrainsCoachType[];
    } {
        if (this.checkIsDefaultValue(trainTariffClass)) {
            return {};
        }

        return {trainTariffClass};
    }
}

export default TrainTariffClassFilterManager;
