import {
    ETrainsFilterType,
    ITrainsFilters,
    TTrainsFilter,
} from 'types/trains/search/filters/ITrainsFilters';
import {
    ITrainsFilteredVariantsInfo,
    TTrainsVariantId,
    TTrainsVariantIdsByFilterType,
    TTrainsVariantWithVisibleStatusById,
} from 'types/trains/common/variant/ITrainsVariant';
import {TrainsSearchContextType} from 'reducers/trains/context/types';

import {ITrainsSearchReducer} from 'reducers/trains/genericSearch/search/reducer';

import {TRAINS_FILTER_MANAGER_TYPES} from 'projects/trains/lib/genericSearch/filters/managers';
import {getFilterManagerByType} from 'projects/trains/lib/genericSearch/filters/managers/getFilterManagerByType';
import {getTrainsFilteredVariantsWithFilterTypesByManagerTypes} from 'projects/trains/lib/genericSearch/filters/getTrainsFilteredVariantsInfo';

/*
 * Фильтруем варианты и тарифы только по цене, после этого полученные тарифы используем для фильтра по типу вагона
 * */

export const getTrainsVariantWithVisibleStatusByIdAfterApplyPriceFilter = ({
    context,
    filters,
    originalSearchInfo,
    originalVariantWithVisibleStatusById,
}: {
    filters: ITrainsFilters;
    context: TrainsSearchContextType;
    originalSearchInfo: ITrainsSearchReducer;
    originalVariantWithVisibleStatusById: TTrainsVariantWithVisibleStatusById;
}): TTrainsVariantWithVisibleStatusById => {
    const {variants} = originalSearchInfo;
    const filterManager = getFilterManagerByType(ETrainsFilterType.PRICE_RANGE);
    const filterInfo = filters[ETrainsFilterType.PRICE_RANGE];
    const isActiveFilter = filterManager?.checkActiveValue(
        filterInfo.value as any,
    );

    if (!isActiveFilter) {
        return originalVariantWithVisibleStatusById;
    }

    const {variantWithVisibleStatusById} =
        getTrainsFilteredVariantsWithFilterTypesByManagerTypes({
            filters,
            context,
            variants,
            filterManagerTypes: [ETrainsFilterType.PRICE_RANGE],
        });

    return variantWithVisibleStatusById;
};

/*
 * Для каждого фильтра определяем те элементы в группе, которые не приведут к пустой выдаче.
 * Для каждого элемента группы фильтра получаем минимальную цену варианта.
 * Для фильтра по тарифам используем только те тарифы, которые получены после применения только фильтра по цене.
 * */

const getFilterActiveOptionsAndOptionMinPriceList = ({
    type,
    filters,
    context,
    filteredVariantIds,
    visibleVariantIdsByFilterType,
    variantWithVisibleStatusById,
    variantWithVisibleStatusByIdAfterApplyPriceFilter,
}: {
    type: ETrainsFilterType;
    filters: ITrainsFilters;
    context: TrainsSearchContextType;
    filteredVariantIds: TTrainsVariantId[];
    variantWithVisibleStatusById: TTrainsVariantWithVisibleStatusById;
    visibleVariantIdsByFilterType: TTrainsVariantIdsByFilterType | null;
    variantWithVisibleStatusByIdAfterApplyPriceFilter: TTrainsVariantWithVisibleStatusById;
}): TTrainsFilter => {
    const filter = filters[type];
    const {options} = filter;
    const filterManager = getFilterManagerByType(type);
    const variantIds =
        visibleVariantIdsByFilterType?.[type] || filteredVariantIds;

    const filterVariants = variantIds.map(variantId => {
        if (type === ETrainsFilterType.TRAIN_TARIFF_CLASS) {
            return variantWithVisibleStatusByIdAfterApplyPriceFilter[variantId]
                .variant;
        }

        return variantWithVisibleStatusById[variantId]?.variant;
    });

    // trains "any" type for filterManager + filter
    const {activeOptions, optionMinPriceList} =
        filterManager.calculateActiveOptionsAndMinPrices({
            variants: filterVariants,
            options: options as any,
            context,
        });

    // trains "any" type for filterManager + filter
    return {
        ...filter,
        optionMinPriceList,
        activeOptions: activeOptions as any,
    };
};

/*
 * Для каждого фильтра обновляем его список доступных элементов в группе и минимальную цену варианта после его применения.
 * */

export const updateTrainsFilterActiveOptionsAndMinPrices = ({
    context,
    filters,
    originalSearchInfo,
    filteredVariantsInfo,
    originalVariantWithVisibleStatusById,
}: {
    context: TrainsSearchContextType;
    originalSearchInfo: ITrainsSearchReducer;
    filters: ITrainsFilters | null;
    filteredVariantsInfo: ITrainsFilteredVariantsInfo;
    originalVariantWithVisibleStatusById: TTrainsVariantWithVisibleStatusById;
}): ITrainsFilters | null => {
    if (!filters) {
        return null;
    }

    const {variantWithVisibleStatusById, visibleVariantIdsByFilterType} =
        filteredVariantsInfo;
    const variantWithVisibleStatusByIdAfterApplyPriceFilter =
        getTrainsVariantWithVisibleStatusByIdAfterApplyPriceFilter({
            context,
            filters,
            originalSearchInfo,
            originalVariantWithVisibleStatusById,
        });
    const filteredVariantIds = originalSearchInfo.variants
        .filter(variant => variantWithVisibleStatusById[variant.id]?.isVisible)
        .map(variant => variant.id);

    return TRAINS_FILTER_MANAGER_TYPES.reduce<ITrainsFilters>(
        (resultFilters, type) => {
            // @ts-ignore trains "any" type for filterManager + filter
            resultFilters[type] = getFilterActiveOptionsAndOptionMinPriceList({
                type,
                context,
                filters,
                filteredVariantIds,
                visibleVariantIdsByFilterType,
                variantWithVisibleStatusById,
                variantWithVisibleStatusByIdAfterApplyPriceFilter,
            });

            return resultFilters;
        },
        {} as ITrainsFilters,
    );
};
