import _intersection from 'lodash/intersection';

import {ITrainsSegment} from 'types/trains/common/segment/ITrainsSegment';
import {ETrainsCoachType} from 'types/trains/common/tariffs/ETrainsCoachType';
import {TTrainsMinTariffsClasses} from 'types/trains/common/tariffs/ITrainsMinTariffs';
import {
    ITrainsVariant,
    ITrainsVariantAndDirection,
} from 'types/trains/common/variant/ITrainsVariant';

import {getTrainsVariantSegments} from 'projects/trains/lib/genericSearch/variants/getTrainsVariantSegments';
import {updateTrainsVariantSegments} from 'projects/trains/lib/genericSearch/variants/updateTrainsVariantSegments';
import {getTrainsSegmentTariffClasses} from 'projects/trains/lib/genericSearch/segments/getTrainsSegmentTariffClasses';
import {updateTrainsSegmentWithTariffsClasses} from 'projects/trains/lib/genericSearch/tariffs/updateTrainsSegmentWithTariffsClasses';

export const getTrainsIntersectionSegmentCoachTypes = ({
    segment,
    coachTypes,
}: {
    segment: ITrainsSegment;
    coachTypes: ETrainsCoachType[];
}): ETrainsCoachType[] => {
    const {tariffs} = segment;
    const segmentTariffClasses = getTrainsSegmentTariffClasses(tariffs);
    const segmentTariffCoachTypes = segmentTariffClasses.map(
        tariffClass => tariffClass.type,
    );

    return _intersection(segmentTariffCoachTypes, coachTypes);
};

export const filterTrainsSegmentClassesByCoachTypes = ({
    segment,
    coachTypes,
}: {
    segment: ITrainsSegment;
    coachTypes: ETrainsCoachType[];
}): TTrainsMinTariffsClasses => {
    return coachTypes.reduce<TTrainsMinTariffsClasses>(
        (resultSegmentClasses, coachType) => {
            const tariffClass = segment.tariffs?.classes?.[coachType];

            if (tariffClass) {
                resultSegmentClasses[coachType] = tariffClass;
            }

            return resultSegmentClasses;
        },
        {},
    );
};

export const checkAndFilterTrainsVariantTariffsByCoachTypes = ({
    coachTypes,
    variantAndDirection,
}: {
    coachTypes: ETrainsCoachType[];
    variantAndDirection: ITrainsVariantAndDirection;
}): ITrainsVariant | null => {
    const segments = getTrainsVariantSegments(variantAndDirection);
    const preparedVariantSegments = segments.reduce<ITrainsSegment[] | null>(
        (resultSegments, segment) => {
            if (!resultSegments) {
                return null;
            }

            const intersectionCoachTypes =
                getTrainsIntersectionSegmentCoachTypes({segment, coachTypes});

            if (!intersectionCoachTypes.length) {
                return null;
            }

            const segmentTariffClasses = filterTrainsSegmentClassesByCoachTypes(
                {segment, coachTypes: intersectionCoachTypes},
            );
            const preparedSegment = updateTrainsSegmentWithTariffsClasses({
                segment,
                tariffClasses: segmentTariffClasses,
            });

            resultSegments.push(preparedSegment);

            return resultSegments;
        },
        [],
    );

    if (!preparedVariantSegments) {
        return null;
    }

    return updateTrainsVariantSegments({
        variantAndDirection,
        segments: preparedVariantSegments,
    });
};
