import {ITrainsSegment} from 'types/trains/common/segment/ITrainsSegment';
import {ITrainsSettlement} from 'types/trains/common/settlement/ITrainsSettlement';

import {CHAR_NBSP, CHAR_RIGHT_ARROW} from 'utilities/strings/charCodes';

import * as i18nBlock from 'i18n/trains-segment-transfer';

const getFullTransferDescriptionWithChangedStation = ({
    stationTitle,
    settlement,
}: {
    stationTitle: string;
    settlement: ITrainsSettlement;
}): string => {
    if (settlement.titleLocative && settlement.preposition) {
        return i18nBlock.transferDashChangeDashPointDashDescriptionDashFullDashLocative(
            {
                settlement: settlement.titleLocative,
                settlementPreposition: settlement.preposition,
                station: stationTitle,
            },
        );
    }

    return i18nBlock.transferDashChangeDashPointDashDescriptionDashFullDashInfinitive(
        {
            settlement: settlement.title,
            station: stationTitle,
        },
    );
};

const getFullTransferDescription = ({
    stationTitle,
    settlement,
}: {
    stationTitle: string;
    settlement: ITrainsSettlement;
}): string => {
    if (settlement.titleLocative && settlement.preposition) {
        return i18nBlock.transferDashPointDashDescriptionDashFullDashLocative({
            settlement: settlement.titleLocative,
            settlementPreposition: settlement.preposition,
            station: stationTitle,
        });
    }

    return i18nBlock.transferDashPointDashDescriptionDashFullDashInfinitive({
        settlement: settlement.title,
        station: stationTitle,
    });
};

const getShortTransferDescription = ({
    stationTitle,
    settlement,
}: {
    stationTitle: string;
    settlement?: ITrainsSettlement;
}): string => {
    if (settlement) {
        if (settlement.titleLocative && settlement.preposition) {
            return i18nBlock.transferDashPointDashDescriptionDashShortDashLocative(
                {
                    transferPoint: settlement.titleLocative,
                    preposition: settlement.preposition,
                },
            );
        }

        if (settlement.title) {
            return i18nBlock.transferDashPointDashDescriptionDashShortDashInfinitive(
                {
                    transferPoint: settlement.title,
                },
            );
        }
    }

    return i18nBlock.transferDashPointDashDescriptionDashShortDashInfinitive({
        transferPoint: stationTitle,
    });
};

export const getTrainsTransferTypeAndDescriptionBySegments = ({
    currentSegment,
    nextSegment,
}: {
    currentSegment: ITrainsSegment;
    nextSegment: ITrainsSegment;
}): {
    description: string;
    isStationChanged: boolean;
} => {
    const {stationTo} = currentSegment;
    const settlement = stationTo?.settlement;

    if (settlement) {
        const stationToTitle = stationTo.title;
        const nextStationFrom = nextSegment.stationFrom;

        if (
            stationTo.id === nextStationFrom.id &&
            stationToTitle.includes(settlement.title)
        ) {
            return {
                isStationChanged: false,
                description: getShortTransferDescription({
                    settlement,
                    stationTitle: stationToTitle,
                }),
            };
        }

        if (stationTo.id !== nextStationFrom.id) {
            const nextStationFromTitle = nextStationFrom.title;
            const stationTitleWithNextStationTitle = `${stationToTitle}${CHAR_NBSP}${CHAR_RIGHT_ARROW} ${nextStationFromTitle}`;

            return {
                isStationChanged: true,
                description: getFullTransferDescriptionWithChangedStation({
                    settlement,
                    stationTitle: stationTitleWithNextStationTitle,
                }),
            };
        }

        return {
            isStationChanged: false,
            description: getFullTransferDescription({
                settlement,
                stationTitle: stationToTitle,
            }),
        };
    }

    return {
        isStationChanged: false,
        description: getShortTransferDescription({
            stationTitle: stationTo.title,
        }),
    };
};
