import {WHEN_SPECIAL_VALUE} from 'types/common/When';
import {ITrainsSearchSort} from 'projects/trains/lib/sort/types';
import {INotFoundTitle} from 'projects/trains/lib/meta/types';
import {ITrainsFilledSearchContext} from 'reducers/trains/context/types';

import isAllDaysSearch from 'projects/trains/lib/search/isAllDaysSearch';
import {getRouteTitle} from 'projects/trains/lib/meta/getRouteTitle';
import {prepareQaAttributes} from 'utilities/qaAttributes/qaAttributes';
import getTrainsSortingQuery from 'projects/trains/lib/sort/getTrainsSortingQuery';
import {trainsURLs} from 'projects/trains/lib/urls';
import {insertJSXIntoKey} from 'utilities/tanker/insertJSXIntoKey';
import {getRoutePoints} from 'projects/trains/lib/meta/getRoutePoints/getRoutePoints';
import {getRouteHumanFormatDate} from 'projects/trains/lib/meta/getRouteHumanFormatDate/getRouteHumanFormatDate';

import * as i18nBlock from 'i18n/trains-search-title-notfound';
import * as i18nWhenSpecialValuesBlock from 'i18n/trains-whenSpecialValues';

import Link from 'components/Link/Link';

/*
 * Вернет объект заголовка для страницы выдачи "Ничего не найдено" на конкретную дату
 */
function getDateNotFoundTitle(
    context: ITrainsFilledSearchContext,
    sort: ITrainsSearchSort,
): INotFoundTitle {
    const {titleInflected} = getRouteTitle(context);

    const params = {
        routeTitleInflected: titleInflected,
        date: getRouteHumanFormatDate(context),
    };

    const allDaysLink = trainsURLs.getTrainsSearchUrl({
        context: {
            from: context.from.slug,
            to: context.to.slug,
            when: WHEN_SPECIAL_VALUE.ALL_DAYS,
        },
        sort: getTrainsSortingQuery(sort),
    });

    const allDaysText = i18nWhenSpecialValuesBlock.allDashDays().toLowerCase();

    return {
        title: i18nBlock.titleDashDateDashTrain(params),
        subtitle: insertJSXIntoKey(i18nBlock.subtitleDashDateDashTrainDash2)({
            link: (
                <Link
                    key="allDaysLink"
                    to={allDaysLink}
                    {...prepareQaAttributes('all-days-link')}
                >
                    {allDaysText}
                </Link>
            ),
        }),
    };
}

/*
 * Вернет объект заголовка для страницы выдачи "Ничего не найдено" на все дни
 */
export function getAllDaysNotFoundTitle(
    context: ITrainsFilledSearchContext,
): INotFoundTitle {
    const {titleInflected} = getRouteTitle(context);
    const params = {routeTitleInflected: titleInflected};

    return {
        title: i18nBlock.titleDashAlldaysDashTrain(params),
        subtitle: i18nBlock.subtitleDashAlldaysDashTrain(),
    };
}

export function getNewNotFoundTitle(
    context: ITrainsFilledSearchContext,
): INotFoundTitle {
    const points = getRoutePoints(context);
    const {titleInflected} = getRouteTitle(points);
    const params = {routeTitleInflected: titleInflected};

    return {
        title: i18nBlock.titleDashNotDashFound(params),
        subtitle: i18nBlock.subtitleDashNotDashFound(),
    };
}

/*
 * Вернет объект заголовка для страницы выдачи "Ничего не найдено"
 */
export default function getNotFoundTitle({
    context,
    sort,
}: {
    context: ITrainsFilledSearchContext;
    sort?: ITrainsSearchSort;
}): INotFoundTitle {
    if (isAllDaysSearch(context) || !sort) {
        return getAllDaysNotFoundTitle(context);
    }

    return getDateNotFoundTitle(context, sort);
}
