import {TLD} from 'constants/tld';

import {WHEN_SPECIAL_VALUE} from 'types/common/When';
import {
    ICrumb,
    isNotNullableCrumbs,
} from 'projects/trains/lib/meta/searchMeta/semanticData/types';
import {ITrainsSearchSort} from 'projects/trains/lib/sort/types';
import {ITrainsFilledSearchContext} from 'reducers/trains/context/types';
import {ITrainsFilters} from 'types/trains/search/filters/ITrainsFilters';

import {CHAR_NBSP, CHAR_EM_DASH} from 'utilities/strings/charCodes';
import isAllDaysSearch from 'projects/trains/lib/search/isAllDaysSearch';
import getFiltersQuery from 'projects/trains/lib/filters/getFiltersQuery';
import getHumanDateWithSpecialValues from 'projects/trains/lib/date/getHumanDateWithSpecialValues';
import getTrainsPageCrumb from 'projects/trains/lib/meta/searchMeta/semanticData/getTrainsPageCrumb';
import getTrainsSortingQuery from 'projects/trains/lib/sort/getTrainsSortingQuery';
import {trainsURLs} from 'projects/trains/lib/urls';
import {getRouteDate} from 'projects/trains/lib/meta/getRouteDate/getRouteDate';
import {getRoutePoints} from 'projects/trains/lib/meta/getRoutePoints/getRoutePoints';

import * as i18nBlock from 'i18n/trains-breadcrumbs';

/*
 * Возвращает хлебную крошку поиска поездов на все дни
 */
function getAllDaysTrainPage(
    context: ITrainsFilledSearchContext,
    urlOptions?: {
        withOrigin: true;
        tld?: TLD;
    },
): ICrumb {
    const {from, to} = getRoutePoints(context);

    const points = `${
        from.popularTitle || from.title
    }${CHAR_NBSP}${CHAR_EM_DASH} ${to.popularTitle || to.title}`;

    return {
        url: trainsURLs.getTrainsSearchUrl(
            {
                context: {
                    from: from.slug,
                    to: to.slug,
                    when: WHEN_SPECIAL_VALUE.ALL_DAYS,
                },
            },
            urlOptions,
        ),
        name: i18nBlock.trainsDashAllDashDay({points}),
        title: i18nBlock.trainsDashAllDashDayDashTitle({points}),
    };
}

/*
 * Возвращает хлебную крошку поиска поездов на конкретный день
 */
function getDatePage({
    context,
    sort,
    filters,
    urlOptions,
}: {
    context: ITrainsFilledSearchContext;
    sort: ITrainsSearchSort | undefined;
    filters?: ITrainsFilters | null;
    urlOptions?: {
        withOrigin: true;
        tld?: TLD;
    };
}): ICrumb | null {
    if (isAllDaysSearch(context) || !sort) {
        return null;
    }

    const {from, to} = getRoutePoints(context);
    const routeDate = getRouteDate(context);

    return {
        url: trainsURLs.getTrainsSearchUrl(
            {
                context: {
                    from: from.slug,
                    to: to.slug,
                    when: routeDate,
                },
                filters: getFiltersQuery(filters),
                sort: getTrainsSortingQuery(sort),
            },
            urlOptions,
        ),
        name: i18nBlock.onDashDate({
            date: getHumanDateWithSpecialValues(routeDate).toLowerCase(),
        }),
    };
}

/*
 * Возвращает хлебные крошки для страницы поиска поездов
 */
export default function getCrumbs({
    context,
    filters,
    sort,
    urlOptions,
}: {
    context: ITrainsFilledSearchContext;
    filters?: ITrainsFilters | null;
    sort?: ITrainsSearchSort;
    urlOptions?: {
        withOrigin: true;
        tld?: TLD;
    };
}): ICrumb[] {
    const crumbs = [
        getTrainsPageCrumb(urlOptions),
        getAllDaysTrainPage(context, urlOptions),
        getDatePage({context, sort, filters, urlOptions}),
    ].filter(Boolean);

    if (isNotNullableCrumbs(crumbs)) {
        return crumbs;
    }

    return [];
}
