import mapValues from 'lodash/mapValues';
import minBy from 'lodash/minBy';

import {PLACE_RESERVATION_TYPE} from 'projects/trains/constants/placeReservationType';
import {EPlaceType} from 'projects/trains/lib/segments/tariffs/constants';

import {ITrainsCoach} from 'reducers/trains/order/types';
import {ITrainsSchema} from 'server/api/TrainsApi/types/ITrainsDetailsApiResponse';
import {TCoachPlaceTypesInfo} from 'projects/trains/components/TrainsOrderPage/SimpleSelectorView/types/TCoachesTypeGroups';

import {TYPE_CATEGORIES_MAP} from 'projects/trains/lib/order/placeCategories';
import getCoachInfo from 'projects/trains/lib/order/getCoachInfo';
import getTariffWithBedding from 'projects/trains/lib/order/getTariffWithBedding';
import IPrice from 'utilities/currency/PriceInterface';
import {CurrencyType} from 'utilities/currency/CurrencyType';

/**
 * Функция возвращает количество мест и минимальную цену вагона без разделения по типу мест
 *
 * @param type - тип места
 * @param coach - вагон
 * @param beddingOption - признак использования белья
 */
function getCount(
    type: string,
    coach: ITrainsCoach,
    beddingOption: boolean,
): TCoachPlaceTypesInfo {
    // @ts-ignore
    const {count, prices, areAllPricesEqual} = getCoachInfo(
        coach,
        beddingOption,
    );

    const minCoachPrice = minBy<IPrice>(prices, 'value');

    return {
        [type]: {
            count,
            areAllPricesEqual,
            minPrice: minCoachPrice || {
                value: 0,
                currency: CurrencyType.RUB,
            },
        },
    };
}

/**
 * Функция для переданного вагона возвращает данные по количеству мест и минимальной цене для кадого типа (верхние, нижние и тд) мест
 *
 * @param coach - вагон
 * @param schema - схема вагона
 * @param beddingOption - признак использования белья
 * @param placeReservationType - тип выкупа мест в данных вагонах
 *
 * Пример возвращаемой структуры: { bottom: { count: 3, minPrice: { value: 1300, currency: 'RUB' } }, top: {...}, ... }
 */
export default function getCoachPlaceCountsAndPricesByPlaceType({
    coach,
    schema,
    beddingOption,
    placeReservationType,
}: {
    coach: ITrainsCoach;
    schema: ITrainsSchema | null;
    beddingOption: boolean;
    placeReservationType: PLACE_RESERVATION_TYPE;
}): TCoachPlaceTypesInfo {
    const {places, type} = coach;

    // получаем типы мест вагона с флагами
    const placeTypesFlagsMap = TYPE_CATEGORIES_MAP[type];

    if (placeReservationType !== PLACE_RESERVATION_TYPE.USUAL) {
        return getCount(EPlaceType.COMPARTMENT, coach, beddingOption);
    }

    // если нет схемы вагона или нет типов мест с флагами для данного типа вагона, считаем все места обычными (regular)
    if (!schema || !placeTypesFlagsMap) {
        return getCount(EPlaceType.REGULAR, coach, beddingOption);
    }

    // флаги из схемы. В них хранятся места для каждого типа мест. Задаются в админке
    const {placeFlags} = schema;

    // проходимся по всем типам мест для вагона (bottom, top и ид)
    // для каждого типа мест placeTypeFlagsMap это объект с флагами
    // Например для bottom: { side: false, upper: false } означает, что место должно быть не сбоку и не верхнее
    return mapValues(placeTypesFlagsMap, placeTypeFlagsMap => {
        const placesAndPrices = {
            count: 0,
            areAllPricesEqual: true,
            minPrice: {
                value: Infinity,
                currency: CurrencyType.RUB,
            },
        };

        // флаги актуальны, если для всех флагов выполняется как минимум одно из условий:
        // 1. значение флага false, т.к. такого флага может не быть в админке
        // и мы будем считать данное место подходящим к типу
        // 2. значение флага true и данный флаг есть админке
        const flagsMapIsActual = Object.keys(placeTypeFlagsMap).every(
            flagName =>
                !placeTypeFlagsMap[flagName] ||
                Boolean(placeFlags[flagName as keyof typeof placeFlags]),
        );

        if (!flagsMapIsActual) {
            return placesAndPrices;
        }

        // проходим по всем местам и проверяем подходит ли оно к текущему типу
        places.forEach(place => {
            const {number, adultTariff} = place;

            // место подходит, если для каждого значения флага (пр. { side: false, upper: false }),
            // выполняется одно из условий:
            // 1. флага в админке нет и значение флага false
            // 2. флаг в админке есть, требование по флагу true и место есть в соответствующем списке
            // 3. флаг в админке есть, требование по флагу false и места в соответствующем списке нет
            const isAppropriatePlace = Object.entries(placeTypeFlagsMap).every(
                ([flagName, requirementFlagValue]) =>
                    (Boolean(placeFlags[flagName as keyof typeof placeFlags]) &&
                        (
                            placeFlags[
                                flagName as keyof typeof placeFlags
                            ] as number[]
                        ).includes(number)) === requirementFlagValue,
            );

            if (isAppropriatePlace) {
                placesAndPrices.count++;

                const placePrice = getTariffWithBedding(
                    adultTariff,
                    coach,
                    beddingOption,
                );

                // записываем информацию, все ли цены в вагоне одинаковые
                placesAndPrices.areAllPricesEqual =
                    placesAndPrices.areAllPricesEqual &&
                    (placesAndPrices.minPrice.value === Infinity ||
                        placesAndPrices.minPrice.value === placePrice.value);

                placesAndPrices.minPrice =
                    minBy([placePrice, placesAndPrices.minPrice], 'value') ||
                    placesAndPrices.minPrice;
            }
        });

        return placesAndPrices;
    });
}
