import uniq from 'lodash/uniq';

import {TRAIN_TARIFF_CLASSES} from 'projects/trains/lib/segments/tariffs/constants';
import {isTrainsCoachType} from 'projects/trains/constants/coachType';

import {ITrainsCoach} from 'reducers/trains/order/types';
import {TCoachesClassesSimpleGroup} from 'projects/trains/components/TrainsOrderPage/SimpleSelectorView/types/TCoachesTypeGroups';

import getCoachClassKey from 'projects/trains/lib/order/getCoachClassKey';
import getDescription from 'projects/trains/lib/order/getDescription';
import getCoachClassAbilities from 'projects/trains/lib/order/getCoachClassAbilities';

import * as i18nBlock from 'i18n/trains-common';

function getClassTitle(coach: ITrainsCoach): string {
    const {
        category,
        serviceClass: {code, title},
    } = coach;
    const isCoachCommon =
        isTrainsCoachType(category.value) &&
        TRAIN_TARIFF_CLASSES.includes(category.value);

    if (isCoachCommon) {
        return i18nBlock._class({code});
    }

    return title ? title : i18nBlock._class({code});
}

/**
 * Функция группирует вагоны по классам обслуживания (ЗЛ, Эконом и тд)
 */
export default function getCoachesGroupedByClasses({
    coaches,
}: {
    coaches: ITrainsCoach[];
}): TCoachesClassesSimpleGroup {
    return coaches.reduce<TCoachesClassesSimpleGroup>((classes, coach) => {
        const {category, facilities = [], serviceClass, owner} = coach;
        const {value: categoryValue} = category;
        const {code, title} = serviceClass;
        const key = getCoachClassKey({coach});

        // title нужен для отображения в плашке класса вагона (например в CoachTypeGroupItem)
        // для обычных вагонов title берем из serviceClass.code, для остальных (Сапсан, Невский экспресс и тд)
        // из serviceClass.title и serviceClass.code
        const isCoachCommon =
            isTrainsCoachType(categoryValue) &&
            TRAIN_TARIFF_CLASSES.includes(categoryValue);
        const classTitle: string = getClassTitle(coach);
        const description = getDescription(serviceClass);

        if (!classes[key]) {
            classes[key] = {
                key,
                code,
                title: classTitle,
                description:
                    !isCoachCommon && description === title
                        ? ''
                        : description.replace(/\.?\n/g, '. '),
                owner,
                coaches: [],
                abilities: getCoachClassAbilities({coach}),
                facilities: [],
            };
        }

        const classesGroup = classes[key];

        if (!classesGroup) {
            return classes;
        }

        classesGroup.coaches.push(coach);
        classesGroup.facilities = uniq([
            ...classesGroup.facilities,
            ...facilities,
        ]);

        return classes;
    }, {});
}
