import mergeWith from 'lodash/mergeWith';

import {PLACE_RESERVATION_TYPE} from 'projects/trains/constants/placeReservationType';
import {isTrainsCoachType} from 'projects/trains/constants/coachType';

import {ITrainsCoach} from 'reducers/trains/order/types';
import {ITrainsSchema} from 'server/api/TrainsApi/types/ITrainsDetailsApiResponse';
import {isNotUndefined} from 'types/utilities';
import {
    ICoachesClass,
    ICoachesTypeGroup,
    TCoachesClassesGroup,
    TCoachesTypeGroups,
} from 'projects/trains/components/TrainsOrderPage/SimpleSelectorView/types/TCoachesTypeGroups';

import setSidePlacesAndPrices from 'projects/trains/lib/order/coaches/setSidePlacesAndPrices';
import {makeCacheable} from 'projects/trains/lib/cache';
import getCoachesGroupedByClasses from 'projects/trains/lib/order/coaches/getCoachesGroupedByClasses';
import getCoachesPlaceCountsAndPricesByPlaceType from 'projects/trains/lib/order/coaches/getCoachesPlaceCountsAndPricesByPlaceType';
import getCoachesGroupedByType from 'projects/trains/lib/order/coaches/getCoachesGroupedByType';
import getBeddingTariff from 'projects/trains/lib/order/getBeddingTariff';

/**
 * Функция возвращает минимальную цену по классу вагона
 * Всегда учитываем стоимость постельного белья при сортировке - https://st.yandex-team.ru/TRAINS-1187
 *
 * @param coaches - вагоны
 * @param placesAndPrices - места и цены класса, разбитые по типам мест
 * @param beddingHasAlreadyIncluded - включено ли уже постельное белье в цену
 */
export function getClassMinPrice({
    coaches,
    placesAndPrices,
    beddingHasAlreadyIncluded,
}: ICoachesClass & {beddingHasAlreadyIncluded: boolean}): number {
    if (!placesAndPrices) {
        return Infinity;
    }

    const beddingTariff = beddingHasAlreadyIncluded
        ? 0
        : Math.max(...coaches.map(getBeddingTariff));

    return Object.values(placesAndPrices).reduce(
        (minClassPrice, iPlacesAndPrices) =>
            iPlacesAndPrices?.minPrice
                ? Math.min(
                      iPlacesAndPrices?.minPrice.value + beddingTariff,
                      minClassPrice,
                  )
                : 0,
        Infinity,
    );
}

function getAggregatedPlacesInfo(
    classes: ICoachesClass[],
): Record<string, {count: number}> | null {
    return classes.reduce<Record<string, {count: number}> | null>(
        (acc, item) =>
            mergeWith(acc, item.placesAndPrices, (obj, src) => {
                return obj && src
                    ? Object.keys(src).reduce(
                          (merged, key) => ({
                              ...merged,
                              [key]: obj.count + src.count,
                          }),
                          {},
                      )
                    : obj || src;
            }),
        null,
    );
}

/**
 * Возвращает тип выкупа мест в данном классе (обычный или всё купе целиком).
 * Если во всех вагонах совпадает тип выкупа мест, то возвращаем его.
 * Если отличается - возвращаем обычный тип выкупа мест.
 */
function getClassPlaceReservationType(
    classCoaches: ITrainsCoach[],
): PLACE_RESERVATION_TYPE {
    const firstCoach = classCoaches[0] as typeof classCoaches[0] | undefined;

    if (!firstCoach) {
        return PLACE_RESERVATION_TYPE.USUAL;
    }

    const firstPlaceReservationType = firstCoach.placeReservationType;

    if (firstPlaceReservationType === PLACE_RESERVATION_TYPE.USUAL) {
        return PLACE_RESERVATION_TYPE.USUAL;
    }

    if (
        classCoaches.some(
            coach => coach.placeReservationType !== firstPlaceReservationType,
        )
    ) {
        return PLACE_RESERVATION_TYPE.USUAL;
    }

    return firstPlaceReservationType;
}

/**
 * Общая функция для группировки вагонов по типам (плацкарт, купе и тд) и дополнительных преобразований этих групп.
 * Внутри группы вагонов по типу, группирует вагоны по классам (1B, эконом и тд) вагонов.
 * Для каждого класса задается количество мест по их типу (боковые, верхние и тд) и минимальная цена для каждого типа
 *
 * Пример возвращаемой структуры: [{ title: platzkarte, classes: [...] }, ...]
 *
 * @param coaches - вагоны
 * @param schemas - схемы вагонов
 * @param beddingOption - признак необходимости постельного белья
 * @returns - отсортированные по минимальной цене группы вагонов
 */
function getCoachesGroupedByTypeWithInfoWithoutCache({
    coaches,
    schemas,
    beddingOption,
}: {
    coaches: ITrainsCoach[];
    schemas: Record<number, ITrainsSchema>;
    beddingOption: boolean;
}): ICoachesTypeGroup[] {
    // Группируем вагоны по типу
    const groups = getCoachesGroupedByType(coaches);
    const groupsWithClasses = Object.entries(groups).reduce<TCoachesTypeGroups>(
        (accGroupsWithClasses, [coachType, coachTypeGroup]) => {
            if (!coachTypeGroup || !isTrainsCoachType(coachType)) {
                return accGroupsWithClasses;
            }

            // берем вагоны из группы и группируем их по классам
            const classes = getCoachesGroupedByClasses({
                coaches: coachTypeGroup.coaches,
            });

            const classesWithPlacesAndPrices = Object.entries(
                classes,
            ).reduce<TCoachesClassesGroup>(
                (accClasses, [classKey, coachClass]) => {
                    if (!coachClass) {
                        return accClasses;
                    }

                    const {coaches: classCoaches} = coachClass;
                    const placeReservationType =
                        getClassPlaceReservationType(classCoaches);

                    // для вагонов класса получаем данные по количеству мест и минимальной цене для кадого типа мест
                    // и объединяем боковые места
                    accClasses[classKey] = {
                        ...coachClass,
                        placesAndPrices: setSidePlacesAndPrices(
                            getCoachesPlaceCountsAndPricesByPlaceType({
                                coaches: classCoaches,
                                schemas,
                                beddingOption,
                                placeReservationType,
                            }),
                        ),
                        placeReservationType:
                            getClassPlaceReservationType(classCoaches),
                    };

                    return accClasses;
                },
                {},
            );

            // записываем в группу отсортированные по минимальной цене классы вагонов
            const typeGroupWithClasses = {
                ...coachTypeGroup,
                classes: Object.values(classesWithPlacesAndPrices)
                    .filter(isNotUndefined)
                    .sort((classOne, classTwo) => {
                        const firstMinPrice = getClassMinPrice({
                            ...classOne,
                            beddingHasAlreadyIncluded: beddingOption,
                        });
                        const secondMinPrice = getClassMinPrice({
                            ...classTwo,
                            beddingHasAlreadyIncluded: beddingOption,
                        });

                        return firstMinPrice - secondMinPrice;
                    }),
            };

            // Получаем агрегированную информацию по всем классам
            const aggregatedPlacesAndPrices = getAggregatedPlacesInfo(
                typeGroupWithClasses.classes,
            );

            if (aggregatedPlacesAndPrices) {
                // Если во всех классах отсутствует какой-либо тип мест - убираем его
                Object.keys(aggregatedPlacesAndPrices).forEach(placesType => {
                    if (!aggregatedPlacesAndPrices[placesType].count) {
                        typeGroupWithClasses.classes.forEach(coachClass => {
                            delete coachClass.placesAndPrices?.[placesType];
                        });
                    }
                });
            }

            accGroupsWithClasses[coachType] = typeGroupWithClasses;

            return accGroupsWithClasses;
        },
        {},
    );

    // Возвращаем отсортированные по минимальной цене группы вагонов
    return Object.values(groupsWithClasses)
        .filter(isNotUndefined)
        .sort(
            ({classes: classesOne}, {classes: classesTwo}) =>
                getClassMinPrice({
                    ...classesOne[0],
                    beddingHasAlreadyIncluded: beddingOption,
                }) -
                getClassMinPrice({
                    ...classesTwo[0],
                    beddingHasAlreadyIncluded: beddingOption,
                }),
        );
}

export default makeCacheable(getCoachesGroupedByTypeWithInfoWithoutCache);
