import minBy from 'lodash/minBy';

import {PLACE_RESERVATION_TYPE} from 'projects/trains/constants/placeReservationType';

import {ITrainsCoach} from 'reducers/trains/order/types';
import {ITrainsSchema} from 'server/api/TrainsApi/types/ITrainsDetailsApiResponse';
import {TCoachPlaceTypesInfo} from 'projects/trains/components/TrainsOrderPage/SimpleSelectorView/types/TCoachesTypeGroups';

import getSchemaByCoach from 'projects/trains/lib/order/getSchemaByCoach';
import getCoachPlaceCountsAndPricesByPlaceType from 'projects/trains/lib/order/coaches/getCoachPlaceCountsAndPricesByPlaceType';
import {CurrencyType} from 'utilities/currency/CurrencyType';

/**
 * Функция для переданных вагонов возвращает данные по количеству мест и минимальной цене для кадого типа(верхние, нижние и тд) мест
 *
 * @param coaches - вагоны
 * @param schemas - схемы вагонов
 * @param beddingOption - признак использования белья
 * @param placeReservationType - тип выкупа мест в данных вагонах
 *
 * Пример возвращаемой структуры: { bottom: { count: 12, minPrice: { value: 1200, currency: 'RUB' } }, top: {...}, ... }
 */
export default function getCoachesPlaceCountsAndPricesByPlaceType({
    coaches,
    schemas,
    beddingOption,
    placeReservationType,
}: {
    coaches: ITrainsCoach[];
    schemas: Record<number, ITrainsSchema>;
    beddingOption: boolean;
    placeReservationType: PLACE_RESERVATION_TYPE;
}): TCoachPlaceTypesInfo | null {
    return coaches.reduce<TCoachPlaceTypesInfo | null>(
        (placesAndPricesByPlaceType, coach) => {
            // получаем данные по количеству мест и цен для конкретного вагона
            const coachPlaceCountsAndPricesByPlaceType =
                getCoachPlaceCountsAndPricesByPlaceType({
                    coach,
                    schema: getSchemaByCoach(coach, schemas),
                    beddingOption,
                    placeReservationType,
                });

            // записываем структуру с местами и ценами для первого вагона
            if (!placesAndPricesByPlaceType) {
                return coachPlaceCountsAndPricesByPlaceType;
            }

            // Итерируемся по текущему вагону, записываем в объект с общей статистикой
            Object.entries(coachPlaceCountsAndPricesByPlaceType).forEach(
                ([placeTypeName, placesAndPrices]) => {
                    if (!placesAndPrices) {
                        return;
                    }

                    const placeTypePlacesAndPrices =
                        placesAndPricesByPlaceType[placeTypeName];

                    // если информация о типе мест была - дополняем, нет - записываем
                    if (placeTypePlacesAndPrices) {
                        const {count, minPrice, areAllPricesEqual} =
                            placesAndPrices;

                        placeTypePlacesAndPrices.count += count;
                        placeTypePlacesAndPrices.areAllPricesEqual =
                            placeTypePlacesAndPrices.areAllPricesEqual &&
                            areAllPricesEqual &&
                            minPrice.value ===
                                placeTypePlacesAndPrices.minPrice.value;
                        placeTypePlacesAndPrices.minPrice = minBy(
                            [minPrice, placeTypePlacesAndPrices.minPrice],
                            'value',
                        ) || {
                            value: 0,
                            currency: CurrencyType.RUB,
                        };
                    } else {
                        placesAndPricesByPlaceType[placeTypeName] =
                            placesAndPrices;
                    }
                },
            );

            return placesAndPricesByPlaceType;
        },
        null,
    );
}
