import _flatMap from 'lodash/flatMap';

import {ITrainGenericService} from 'server/api/GenericOrderApi/types/common/TGenericService';
import {
    ITrainCoachInfo,
    ITrainSegment,
} from 'projects/trains/components/TrainsOrderSegments/types';
import {
    ITrainFeature,
    TRAIN_FEATURES_TYPES,
} from 'projects/trains/lib/segments/features/types';
import ITrainPassenger from 'server/api/GenericOrderApi/types/common/service/ITrainServiceInfo/ITrainPassenger/ITrainPassenger';
import {ETrainsCoachType} from 'types/trains/common/tariffs/ETrainsCoachType';

import {checkHasNonRefundablePassengerTariff} from 'projects/trains/lib/order/checkNonRefundablePassengerTariff';
import {getIsRemoteCheckinEnabledForTicket} from 'projects/account/pages/Order/utilities/getIsRemoteCheckinEnabled';

import * as i18nBlock from 'i18n/account-OrderTrains-Info';

const trimZeroAtFirstPosition = (number: string): string =>
    number.replace(/^0+/, '');

const calculateSegmentPlacesByOrderPassengers = (
    passengers: ITrainPassenger[],
): string[] => {
    return _flatMap<ITrainPassenger, string>(passengers, passenger =>
        (passenger.ticket?.places ?? []).map(({number}) =>
            trimZeroAtFirstPosition(number),
        ),
    );
};

const getSegmentCoachInfo = ({
    carNumber,
    carType,
    passengers,
}: {
    carNumber: string;
    carType: ETrainsCoachType;
    passengers: ITrainPassenger[];
}): ITrainCoachInfo => {
    return {
        type: carType,
        number: trimZeroAtFirstPosition(carNumber),
        places: calculateSegmentPlacesByOrderPassengers(passengers),
    };
};

const getSegmentFeatures = (service: ITrainGenericService): ITrainFeature[] => {
    const {
        trainInfo: {companyTitle, trainInfo},
    } = service;
    const features = [
        {
            type: TRAIN_FEATURES_TYPES.COMPANY,
            content: companyTitle || i18nBlock.company(),
        },
    ];

    if (trainInfo.brandTitle) {
        features.push({
            type: TRAIN_FEATURES_TYPES.DELUXE_TRAIN,
            content: trainInfo.brandTitle,
        });
    }

    return features;
};

export const convertGenericOrderServiceToTrainSegment = (
    service: ITrainGenericService,
): ITrainSegment => {
    const {
        trainInfo: {
            arrival,
            departure,
            stationFrom,
            stationTo,
            carNumber,
            carType,
            trainInfo,
            passengers,
        },
    } = service;

    const features = getSegmentFeatures(service);
    const coachInfo = getSegmentCoachInfo({carNumber, carType, passengers});
    const hasNonRefundableTariff = checkHasNonRefundablePassengerTariff({
        passengers,
    });

    return {
        arrival,
        features,
        departure,
        coachInfo,
        stationFrom,
        stationTo,
        hasNonRefundableTariff,
        title: trainInfo.trainTitle,
        number: trainInfo.trainNumber,
        eTicketStatus: service.trainInfo.passengers.every(
            ({ticket}) => ticket && getIsRemoteCheckinEnabledForTicket(ticket),
        ),
    };
};
