import uniqBy from 'lodash/uniqBy';
import flatten from 'lodash/flatten';

import {GENDER_TYPE} from 'projects/trains/constants/genders';

import {
    ISchemaPlaceFlags,
    ITrainsSchema,
} from 'server/api/TrainsApi/types/ITrainsDetailsApiResponse';
import {ITrainsCoach, TrainsPassengersCount} from 'reducers/trains/order/types';

import {countPassengersWithPlaces} from 'projects/trains/lib/order/passengers/utils';

enum EPlaceStorey {
    LOWER = 'lower',
    UPPER = 'upper',
}

interface ISchemaPlace {
    number: number;
    storey: EPlaceStorey;
    nextPlaces: Set<ISchemaPlace>;
    previousPlaces: Set<ISchemaPlace>;
    groupNumber: number;
    index: number;
}

function getPlaceStorey(
    placeNumber: number,
    placeFlags: ISchemaPlaceFlags,
): EPlaceStorey {
    // middle - это средние места в трехместных купе РИЦ. Пример поезда: СПб - Вена.
    // При наличии средних мест в вагоне все места считаем "нижними", получится простой алгоритм, как в сидячих вагонах.
    // Так сделано, потому что РЖД не умеет принимать запросы с указанным количеством средних мест.
    if (placeFlags.middle?.length > 0) {
        return EPlaceStorey.LOWER;
    }

    const upperPlaceNumbers = placeFlags.upper || [];

    return upperPlaceNumbers.includes(placeNumber)
        ? EPlaceStorey.UPPER
        : EPlaceStorey.LOWER;
}

/**
 * Помечает два места как "смежные". Отонощшение смежности - симметричное.
 */
function connectPlaces(place1: ISchemaPlace, place2: ISchemaPlace): void {
    place1.nextPlaces.add(place2);
    place2.previousPlaces.add(place1);
}

/**
 * Инициализирует граф смежных мест.
 *
 * @param places - Список всех свободных мест.
 */
function fillNextPlaces(places: ISchemaPlace[]): void {
    places.slice(0, places.length - 1).forEach((place, placeIndex) => {
        let nextPlaceOfSameLevel: ISchemaPlace | null = null;
        let nextPlaceOfAnotherLevel: ISchemaPlace | null = null;
        const nextPlaces = places.slice(placeIndex + 1);

        nextPlaces.some(nextPlace => {
            if (!nextPlaceOfSameLevel && place.storey === nextPlace.storey) {
                nextPlaceOfSameLevel = nextPlace;
                connectPlaces(place, nextPlace);
            }

            if (!nextPlaceOfAnotherLevel && place.storey !== nextPlace.storey) {
                nextPlaceOfAnotherLevel = nextPlace;
                connectPlaces(place, nextPlace);
            }

            return nextPlaceOfSameLevel && nextPlaceOfAnotherLevel;
        });
    });
}

/**
 * Формирует список мест с дополнительными свойствами:
 * - index - индекс места в массиве всех свободных мест в вагоне,
 * - storey - ярус места: lower / upper,
 * - nextPlaces - массив смежных мест.
 * @param schema - Схема вагона.
 * @param coach - Выбранный пользователем вагон.
 * @param gender - гендерный признак вагона
 * @returns Список свободных мест в вагоне.
 */
function getPlaces(
    schema: ITrainsSchema,
    coach: ITrainsCoach,
    gender: GENDER_TYPE | null,
): ISchemaPlace[] {
    const places: ISchemaPlace[] = coach.places
        .filter(
            place =>
                !gender ||
                place.gender === gender ||
                place.gender === GENDER_TYPE.SINGLE,
        )
        .map(place => ({
            number: place.number,
            storey: getPlaceStorey(place.number, schema.placeFlags),
            nextPlaces: new Set<ISchemaPlace>(),
            previousPlaces: new Set<ISchemaPlace>(),
            groupNumber: place.groupNumber || 0,
        }))
        .sort(
            (place1, place2) =>
                place1.groupNumber * 100 +
                place1.number -
                place2.groupNumber * 100 -
                place2.number,
        )
        .map((place, index) => ({
            ...place,
            index,
        }));

    fillNextPlaces(places);

    return places;
}

function getVariantKey(variantPlaces: ISchemaPlace[]): string {
    return variantPlaces.map(place => place.number).join('-');
}

/**
 * Генерирует варианты на основе текущего варианта, добавляя следующее место.
 * Пример. Текущий вариант - [2, 3]. Конфигурация вагона такова, что в нем следующие пары смежных мест:
 * (2, 3), (3, 4), (3, 7), (4, 6)
 * В результате получатся новые варианты: [[2, 3, 4], [2, 3, 7]].
 *
 * @param currentVariantPlaces - массив мест, входящих в текущий вариант.
 *
 * @returns Массив новых вариантов.
 */
function getNextVariants(
    currentVariantPlaces: ISchemaPlace[],
): ISchemaPlace[][] {
    const lastPlace = currentVariantPlaces[currentVariantPlaces.length - 1];

    return [...lastPlace.nextPlaces].map(place => [
        ...currentVariantPlaces,
        place,
    ]);
}

/**
 * Генерирует варианты на основе текущего варианта, добавляя предшествующее место в начало списка.
 * Пример. Текущий вариант - [3, 4]. Конфигурация вагона такова, что в нем следующие пары смежных мест:
 * (1, 3), (2, 3), (3, 4), (3, 7), (4, 6)
 * В результате получатся новые варианты: [[1, 3, 4], [2, 3, 4]].
 *
 * @param currentVariantPlaces - массив мест, входящих в текущий вариант.
 *
 * @returns массив новых вариантов.
 */
function getPreviousVariants(
    currentVariantPlaces: ISchemaPlace[],
): ISchemaPlace[][] {
    const firstPlace = currentVariantPlaces[0];

    return [...firstPlace.previousPlaces].map(place => [
        place,
        ...currentVariantPlaces,
    ]);
}

/**
 * Генерирует все вомзожные варианты сочетания соседних мест,
 * таких, чтобы в каждом из сочетаний присутствовало первое из уже выбраных пользователем мест
 *
 * @param startPlace - первое место из уже выбранных пользователем мест
 * @param passengersCount - Количество пасажиров
 *
 * @returns массив вариантов. Каждый вариант - массив мест.
 */
function getPlacesVariants(
    startPlace: ISchemaPlace,
    passengersCount: number,
): ISchemaPlace[][] {
    let variants = [[startPlace]];

    for (let stepIndex = 1; stepIndex < passengersCount; stepIndex++) {
        variants = uniqBy(
            flatten([
                ...variants.map(variant => getNextVariants(variant)),
                ...variants.map(variant => getPreviousVariants(variant)),
            ]),
            variant => [getVariantKey(variant), variant],
        );
    }

    return variants;
}

function validateVariant(
    variantPlaces: ISchemaPlace[],
    allPlaces: ISchemaPlace[],
    orderPlaceNumbers: number[],
): boolean {
    if (
        !orderPlaceNumbers.every(orderPlaceNumber =>
            variantPlaces.find(place => place.number === orderPlaceNumber),
        )
    ) {
        return false;
    }

    variantPlaces.sort((place1, place2) => place1.number - place2.number);

    // skippedPlaces - невыбранные места в выбранном диапазоне.
    const skippedPlaces = allPlaces.filter(
        place =>
            place.index > variantPlaces[0].index &&
            place.index < variantPlaces[variantPlaces.length - 1].index &&
            !variantPlaces.includes(place),
    );

    const lowerPlaces = variantPlaces.filter(
        place => place.storey === EPlaceStorey.LOWER,
    );
    const lastLowerPlaceIndex =
        lowerPlaces.length > 0 ? lowerPlaces[lowerPlaces.length - 1].index : -1;
    const upperPlaces = variantPlaces.filter(
        place => place.storey === EPlaceStorey.UPPER,
    );
    const lastUpperPlaceIndex =
        upperPlaces.length > 0 ? upperPlaces[upperPlaces.length - 1].index : -1;

    return skippedPlaces.every(
        place =>
            !(
                (place.storey === EPlaceStorey.UPPER &&
                    place.index < lastUpperPlaceIndex) ||
                (place.storey === EPlaceStorey.LOWER &&
                    place.index < lastLowerPlaceIndex)
            ),
    );
}

/**
 * getAvailablePlaces - возвращает массив доступных для выбора мест, которые будут подсвечены
 *
 * @param schema объект схемы вагона, необходим из-за списка верхних мест
 * @param coach объект текущего вагона, необходим из-за списка всех мест в вагоне
 * @param orderPlaces массив выбранных номеров мест
 * @param passengers объект кол-ва пассажиров по возрастным категориям
 * @param gender гендерный признак вагона
 * @param schemeHasGroupedPlaces в схеме есть объединенные места (для выкупа купе целиком)
 */
export default function getAvailablePlaces(
    schema: ITrainsSchema,
    coach: ITrainsCoach,
    orderPlaces: number[],
    passengers: TrainsPassengersCount,
    gender: GENDER_TYPE | null,
    schemeHasGroupedPlaces: boolean,
): number[] {
    const passengersCount = countPassengersWithPlaces(passengers);

    const places = getPlaces(schema, coach, gender);

    // ничего не выбрано, можно выбрать любое место
    // если 1 пассажир, то давать выбрать любое свободное место
    // если можно выкупить только купе целиком, то даем выбрать любое купе
    if (
        orderPlaces.length === 0 ||
        passengersCount === 1 ||
        schemeHasGroupedPlaces
    ) {
        return places.map(place => place.number);
    }

    const firstOrderPlace = places.find(
        place => place.number === orderPlaces[0],
    );

    if (!firstOrderPlace) {
        return places.map(place => place.number);
    }

    let variants = getPlacesVariants(firstOrderPlace, passengersCount);

    variants = variants.filter(variant =>
        validateVariant(variant, places, orderPlaces),
    );

    const placeNumberSet = new Set(
        flatten([...variants]).map(place => place.number),
    );

    return [...placeNumberSet].sort((number1, number2) => number1 - number2);
}
