import moment from 'moment-timezone';
import partition from 'lodash/partition';
import first from 'lodash/first';
import last from 'lodash/last';

import {ITrainsStation} from 'types/trains/common/station/ITrainsStation';
import ITrainRailwayStation from 'server/api/GenericOrderApi/types/common/service/ITrainServiceInfo/ITrainRailwayStation';
import {ITrainGenericService} from 'server/api/GenericOrderApi/types/common/TGenericService';
import {EDirection} from 'types/common/EDirection';
import ITrainServiceInfo from 'server/api/GenericOrderApi/types/common/service/ITrainServiceInfo/ITrainServiceInfo';

import {IOrderTripInfo} from 'selectors/trains/order/orderTripInfoSelector';

import {formatDate} from 'utilities/dateUtils';
import {ROBOT} from 'utilities/dateUtils/formats';

// You can drop ITrainRailwayStation after support ITrainsSettlement in API
type TTrainStation = ITrainsStation & ITrainRailwayStation;

const getStationSettlementTitle = (
    station: TTrainStation,
): string | undefined => {
    const {settlement} = station;

    if (settlement) {
        return settlement.title;
    }

    return station?.settlementTitle;
};

const getStationTitle = (station: TTrainStation): string =>
    station.popularTitle || station.title;

const checkStationSettlementsEqual = ({
    stationFrom,
    stationTo,
}: {
    stationFrom: TTrainStation;
    stationTo: TTrainStation;
}): boolean => {
    const fromSettlementTitle = getStationSettlementTitle(stationFrom);
    const toSettlementTitle = getStationSettlementTitle(stationTo);

    if (!fromSettlementTitle || !toSettlementTitle) {
        return false;
    }

    return (
        fromSettlementTitle.toLocaleLowerCase() ===
        toSettlementTitle.toLocaleLowerCase()
    );
};

const getDirectionPartTitle = ({
    station,
    areStationSettlementsEqual,
}: {
    station: TTrainStation;
    areStationSettlementsEqual: boolean;
}): string => {
    const settlementTitle = getStationSettlementTitle(station);
    const stationTitle = getStationTitle(station);

    if (!settlementTitle) {
        return stationTitle;
    }

    return areStationSettlementsEqual ? stationTitle : settlementTitle;
};

function getDate(trainInfo: ITrainServiceInfo): string {
    const {departure, stationFrom} = trainInfo;

    const departureMoment = moment(departure).tz(stationFrom.timezone);

    return formatDate(departureMoment, ROBOT);
}

export const getOrderTripInfoByGenericOrderServices = (
    services: ITrainGenericService[],
): IOrderTripInfo | null => {
    const sortedServices = services.sort(
        (firstService, secondService) =>
            firstService.trainInfo.segmentIndex -
            secondService.trainInfo.segmentIndex,
    );

    const [forwardServices, backwardServices] = partition(
        sortedServices,
        ({trainInfo}) => trainInfo.direction === EDirection.FORWARD,
    );

    const firstForwardService = first(forwardServices);
    const lastForwardService = last(forwardServices);

    if (!firstForwardService || !lastForwardService) {
        return null;
    }

    const {stationFrom} = firstForwardService.trainInfo;
    const {stationTo} = lastForwardService.trainInfo;
    const areStationSettlementsEqual = checkStationSettlementsEqual({
        stationFrom,
        stationTo,
    });
    const from = getDirectionPartTitle({
        areStationSettlementsEqual,
        station: stationFrom,
    });
    const to = getDirectionPartTitle({
        areStationSettlementsEqual,
        station: stationTo,
    });

    const firstBackwardService = first(backwardServices);

    return {
        from,
        to,
        when: getDate(firstForwardService.trainInfo),
        returnWhen: firstBackwardService
            ? getDate(firstBackwardService.trainInfo)
            : undefined,
    };
};
