import flatMap from 'lodash/flatMap';
import groupBy from 'lodash/groupBy';
import partition from 'lodash/partition';

import {ITrainGenericService} from 'server/api/GenericOrderApi/types/common/TGenericService';
import ITrainTicket from 'server/api/GenericOrderApi/types/common/service/ITrainServiceInfo/ITrainPassenger/ITrainTicket';
import ITrainServiceInfo from 'server/api/GenericOrderApi/types/common/service/ITrainServiceInfo/ITrainServiceInfo';
import ITrainPassenger from 'server/api/GenericOrderApi/types/common/service/ITrainServiceInfo/ITrainPassenger/ITrainPassenger';
import ITrainInsurance from 'server/api/GenericOrderApi/types/common/service/ITrainServiceInfo/ITrainPassenger/ITrainInsurance';
import {IRefundPartInfo} from 'server/api/GenericOrderApi/types/common/refund/IRefundPartInfo';
import {TrainBookedTariffCode} from 'server/api/TrainsBookingApi/types/ITrainsOrderInfoTicket';

export interface ITicketWithTrainInfo {
    ticket: ITrainTicket | null;
    trainInfo: ITrainServiceInfo;
    insurance: ITrainInsurance | null;
    refundPartInfo: IRefundPartInfo | null;
    nonRefundableTariff: boolean;
    withBaby: boolean;
    babyInsurance: ITrainInsurance | null;
    babyRefundPartInfo: IRefundPartInfo | null;
}

export interface IPassengerWithTickets {
    passenger: ITrainPassenger;
    baby: ITrainPassenger | undefined;
    tickets: ITicketWithTrainInfo[];
}

export const getPassengerWithTicketsListByGenericOrderServices = (
    services: ITrainGenericService[],
): IPassengerWithTickets[] => {
    const flattenPassengers = flatMap(services, service => {
        const {trainInfo} = service;

        const [babies, otherPassengers] = partition(
            trainInfo.passengers,
            passenger =>
                passenger.ticket?.bookedTariffCode ===
                TrainBookedTariffCode.BABY,
        );

        return otherPassengers.map(passenger => {
            const baby = babies.find(
                babyPassenger =>
                    babyPassenger.ticket &&
                    passenger.ticket &&
                    babyPassenger.ticket.blankId === passenger.ticket.blankId,
            );

            const ticketTrainInfo: ITicketWithTrainInfo = {
                insurance: passenger.insurance,
                ticket: passenger.ticket,
                refundPartInfo: passenger.refundPartInfo,
                nonRefundableTariff: passenger.nonRefundableTariff,
                trainInfo,
                withBaby: Boolean(baby),
                babyInsurance: baby?.insurance || null,
                babyRefundPartInfo: baby?.refundPartInfo || null,
            };

            return {
                passenger,
                baby,
                ticketTrainInfo,
            };
        });
    });

    const passengersGroupedByCustomerId = groupBy(
        flattenPassengers,
        ({passenger}) => passenger.customerId,
    );

    return Object.values(passengersGroupedByCustomerId).map(trains => ({
        passenger: trains[0].passenger,
        baby: trains[0].baby,
        tickets: trains.map(({ticketTrainInfo}) => ticketTrainInfo),
    }));
};
