import {GENDER_TYPE} from 'projects/trains/constants/genders';

import {
    ITrainsSchema,
    ITrainsSchemaPlace,
} from 'server/api/TrainsApi/types/ITrainsDetailsApiResponse';
import {ITrainsCoach, TrainsPassengersCount} from 'reducers/trains/order/types';
import {ITrainsCoachPlace} from 'server/services/TrainsService/types/ITrainsDetailsInfoServiceResponse';
import {EPricePalette} from 'types/trains/booking/EPricePalette';

import {countPassengersWithPlaces} from 'projects/trains/lib/order/passengers/utils';
import getAvailablePlaces from 'projects/trains/lib/order/getAvailablePlaces';
import getTariffWithBedding from 'projects/trains/lib/order/getTariffWithBedding';
import {
    AVAILABLE,
    RESERVED,
    SELECTED,
    UNAVAILABLE,
} from 'projects/trains/lib/order/placeMap';
import PriceScale from 'projects/trains/lib/order/priceScale/PriceScale';
import IPrice from 'utilities/currency/PriceInterface';

export interface IPlaceMapItem extends ITrainsSchemaPlace {
    theme: EPricePalette | null;
    tariff: IPrice | null;
    placeState: string;
    gender: GENDER_TYPE | null;
    upper: boolean;
    genderTipUp: boolean;
}

/**
 * Возвращает статус места: выбрано, доступно для выбора, недоступно для выбора, занято
 *
 * @param place - данные места из схемы вагона
 * @param coachPlace - данные соответствующего места из trainDetails
 * @param orderPlaces - массив выбранных мест
 * @param availablePlaces - массив мест доступных для выбора
 * @param active - является ли данный вагон активным (выбранным)
 */
function getPlaceState({
    place,
    coachPlace,
    orderPlaces,
    availablePlaces,
    active,
}: {
    place: ITrainsSchemaPlace;
    coachPlace: ITrainsCoachPlace | undefined;
    orderPlaces: number[];
    availablePlaces: number[];
    active: boolean;
}): string {
    if (coachPlace) {
        if (orderPlaces.includes(place.number)) {
            return SELECTED;
        }

        // если вагон не активный, то не обращаем внимание на "активные места",
        // т.к. для него все места активные и нет выбранных мест, которые бы сужали все места для выбора
        return !active || availablePlaces.includes(place.number)
            ? AVAILABLE
            : UNAVAILABLE;
    }

    return RESERVED;
}

/**
 * Возвращает массив мест со статусами (выбрано, доступно для выбора, недоступно для выбора, занято)
 *
 * @param schema - схема вагона
 * @param coach - данные вагона из trainDetails
 * @param scale - шкала с разбиением мест по цене
 * @param active - является ли данный вагон активным (выбранным)
 * @param orderPlaces - массив выбранных мест
 * @param passengers - данные о количестве пассажиров в заказе
 * @param gender - гендерный признак мест
 * @param beddingOption - признак использования белья
 * @param schemeHasGroupedPlaces в схеме есть объединенные места (для выкупа купе целиком)
 */
export default function getPlaceMap({
    schema,
    coach,
    scale,
    active,
    orderPlaces,
    passengers,
    gender,
    beddingOption,
    schemeHasGroupedPlaces,
}: {
    schema: ITrainsSchema | null;
    coach: ITrainsCoach;
    scale: PriceScale;
    active: boolean;
    orderPlaces: number[];
    passengers: TrainsPassengersCount;
    gender: GENDER_TYPE | null;
    beddingOption: boolean;
    schemeHasGroupedPlaces: boolean;
}): IPlaceMapItem[] {
    if (!schema) {
        return [];
    }

    let availablePlaces: number[] = [];

    if (active) {
        const isToggleMode =
            schemeHasGroupedPlaces ||
            countPassengersWithPlaces(passengers) === 1;

        availablePlaces = getAvailablePlaces(
            schema,
            coach,
            orderPlaces,
            passengers,
            isToggleMode ? null : gender,
            schemeHasGroupedPlaces,
        );
    } else {
        // если вагон не активный, то чистим orderPlaces, т.к. для него не нужно отмечать выбранные места
        orderPlaces = [];
    }

    const {upper = [], genderTipUp = []} = schema.placeFlags;

    return schema.places.map(place => {
        const coachPlace = coach.places.find(
            item => item.number === place.number,
        );
        const tariff =
            (coachPlace &&
                getTariffWithBedding(
                    coachPlace.adultTariff,
                    coach,
                    beddingOption,
                )) ||
            null;

        return {
            ...place,
            theme: scale.getTheme(tariff),
            tariff,
            placeState: getPlaceState({
                place,
                coachPlace,
                orderPlaces,
                availablePlaces,
                active,
            }),
            gender: (coachPlace && coachPlace.gender) || null,
            upper: upper.includes(place.number),
            genderTipUp: genderTipUp.includes(place.number),
        };
    });
}
