import moment, {Moment} from 'moment-timezone';

import {TTrainsStoreOrderSegment} from 'projects/trains/lib/segments/types';
import {ITrainsCoach} from 'reducers/trains/order/types';

import {TIME} from 'utilities/dateUtils/formats';
import {isSameTime} from 'projects/trains/lib/date/utils';
import getRailwayTimeCode from 'projects/trains/lib/date/getRailwayTimeCode';

import * as i18nBlock from 'i18n/trains-through-coach';
import * as i18nTrainsTimezonesBlock from 'i18n/trains-timezones';

/**
 * Возвращает тип, означающий как необходимо выводить время прибытия временем прибытия беспересадочного вагона.
 *
 * time - время.
 * datetime - дата и время.
 * railway-time - время по местному времени и по временной зоне железнодорожного сообщения.
 * railway-datetime - дата и время по местному времени и по временной зоне железнодорожного сообщения.
 *
 * @param throughArrivalMoment - время прибытия беспересадочного вагона в timezone.
 * @param arrivalMoment - время прибытия основного поезда в timezone.
 * @param throughArrivalRailwayMoment - время прибытия беспересадочного вагона в railwayTimezone.
 */
function getThroughCoachArrivalTranslationType({
    throughArrivalMoment,
    arrivalMoment,
    throughArrivalRailwayMoment,
}: {
    throughArrivalMoment: Moment;
    arrivalMoment: Moment;
    throughArrivalRailwayMoment: Moment;
}): string {
    const showDate = throughArrivalMoment.date() !== arrivalMoment.date();

    if (isSameTime(throughArrivalMoment, throughArrivalRailwayMoment)) {
        if (showDate) {
            return 'datetime';
        }

        return 'time';
    }

    if (showDate) {
        return 'railway-datetime';
    }

    return 'railway-time';
}

/**
 * Возвращает переведенную строку с временем прибытия беспересадочного вагона.
 *
 * @param params - параметры функции
 * @param params.segment - сегмент.
 * @param params.coach - беспересадочный вагон.
 * @param [params.plural=false] - признак использования множественного числа.
 */
export function getThroughCoachArrivalTime({
    segment,
    coach,
    plural = false,
}: {
    segment: TTrainsStoreOrderSegment;
    coach: ITrainsCoach;
    plural?: boolean;
}): string {
    const coachArrival = coach.throughArrival || coach.arrival;

    const arrivalMoment = moment.tz(
        segment.arrival,
        segment.stationTo.timezone,
    );
    const throughArrivalMoment = moment.tz(
        coachArrival,
        segment.stationTo.timezone,
    );
    const throughArrivalRailwayMoment = moment.tz(
        coachArrival,
        segment.stationTo.railwayTimezone,
    );
    const type = getThroughCoachArrivalTranslationType({
        throughArrivalMoment,
        arrivalMoment,
        throughArrivalRailwayMoment,
    });
    const railwayTimeCode = getRailwayTimeCode(
        segment.stationTo.railwayTimezone,
    );

    const railwayDate = throughArrivalRailwayMoment.format(
        i18nBlock.dateDashFormat(),
    );

    const railwayDatetime = throughArrivalRailwayMoment.format(
        i18nBlock.datetimeDashFormat(),
    );
    const railwayTime = throughArrivalRailwayMoment.format(TIME);

    const localDate = throughArrivalMoment.format(i18nBlock.dateDashFormat());
    const localTime = throughArrivalMoment.format(TIME);

    const timezone = getTimezoneName(railwayTimeCode, false);

    switch (type) {
        case 'datetime':
            return i18nBlock.arrivalDashDatetime({
                plural,
                railwayDate,
                railwayTime,
            });
        case 'railway-datetime':
            return i18nBlock.arrivalDashRailwayDashDatetime({
                plural,
                localDate,
                localTime,
                railwayDatetime,
                timezone,
            });
        case 'railway-time':
            return i18nBlock.arrivalDashRailwayDashTime({
                plural,
                localTime,
                railwayTime,
                timezone,
            });
        case 'time':
            return i18nBlock.arrivalDashTime({
                plural,
                railwayTime,
            });
        default:
            return '';
    }
}

function getTimezoneName(railwayTimeCode: string, isShort: boolean): string {
    switch (railwayTimeCode) {
        case 'almaty':
            return isShort
                ? i18nTrainsTimezonesBlock.almatyDashShort()
                : i18nTrainsTimezonesBlock.almaty();
        case 'kiev':
            return isShort
                ? i18nTrainsTimezonesBlock.kievDashShort()
                : i18nTrainsTimezonesBlock.kiev();
        case 'minsk':
            return isShort
                ? i18nTrainsTimezonesBlock.minskDashShort()
                : i18nTrainsTimezonesBlock.minsk();
        case 'moscow':
            return isShort
                ? i18nTrainsTimezonesBlock.moscowDashShort()
                : i18nTrainsTimezonesBlock.moscow();
        default:
            return isShort
                ? i18nTrainsTimezonesBlock.localDashTimeDashShort()
                : i18nTrainsTimezonesBlock.localDashTime();
    }
}
