import clone from 'lodash/clone';
import times from 'lodash/times';

import {
    PASSENGERS_TYPE_ORDER,
    PASSENGERS_TYPES,
} from 'projects/trains/constants/passengersTypes';
import {TRAIN_COACH_TYPE} from 'projects/trains/constants/coachType';

import {
    IAdditionalSchemeInfo,
    ITrainsCoach,
    TrainsPassengersCount,
} from 'reducers/trains/order/types';
import {ITrainPriceDetails} from 'projects/trains/lib/order/getTrainPriceDetails/types/ITrainPriceDetails';

import IPrice from 'utilities/currency/PriceInterface';
import getInitialTrainPriceDetails from 'projects/trains/lib/order/getTrainPriceDetails/getInitialTrainPriceDetails';
import extendTrainPriceDetailsByBedding from 'projects/trains/lib/order/getTrainPriceDetails/extendTrainPriceDetailsByBedding';
import getReservationVariantByPassengers from 'projects/trains/lib/order/getTrainPriceDetails/utilities/getReservationVariantByPassengers';

interface IGetApproximatedTrainPriceDetailsParams {
    passengers: TrainsPassengersCount;
    orderPlaces: number[];
    coach: ITrainsCoach | null;
    additionalSchemeInfo: IAdditionalSchemeInfo[];
    nonRefundableTariff: boolean;
    beddingOption: boolean;
}

function getSuitablePriceByCoach<Price extends IPrice | null>(
    type: PASSENGERS_TYPES.ADULTS | PASSENGERS_TYPES.CHILDREN,
    coach: ITrainsCoach,
    adultPrice: Price,
    childPrice: Price,
): Price {
    /**
     * Не применяем детский тариф для вагонов люкс
     * @see https://st.yandex-team.ru/TRAVELFRONT-1258
     */
    if (
        coach.type === TRAIN_COACH_TYPE.SOFT ||
        type === PASSENGERS_TYPES.ADULTS
    ) {
        return clone(adultPrice);
    }

    return clone(childPrice);
}

/**
 * Функция возвращает ориентировочную цену на выбранные места.
 * Считаем до формирования заказа
 * Чтобы выдать минимальную цену, выбираем места с максимальными тарифами для взрослых и садим на них детей.
 * На оставшиеся места садим взрослых.
 */
export default function getApproximatedTrainPriceDetails(
    params: IGetApproximatedTrainPriceDetailsParams,
): ITrainPriceDetails {
    const {
        passengers,
        orderPlaces,
        coach,
        additionalSchemeInfo,
        nonRefundableTariff,
        beddingOption,
    } = params;

    const priceDetails = getInitialTrainPriceDetails({
        tariffs: PASSENGERS_TYPE_ORDER,
        passengers,
        additionalSchemeInfo,
        coach,
        isApproximatePrice: true,
    });

    if (orderPlaces.length === 0 || !coach) {
        return priceDetails;
    }

    const sortedCoachPlaces = coach.places
        .filter(place => orderPlaces.includes(place.number))
        .sort((a, b) => b.adultTariff.value - a.adultTariff.value);

    const baseCurrency = sortedCoachPlaces[0].adultTariff.currency;

    priceDetails.refundablePrice.currency = baseCurrency;
    priceDetails.bedding.currency = baseCurrency;
    priceDetails.coachPlaces = sortedCoachPlaces;

    if (coach.reservationVariants) {
        const reservationVariant = getReservationVariantByPassengers(
            coach.reservationVariants,
            passengers,
        );

        if (!reservationVariant) {
            return priceDetails;
        }

        priceDetails.isPlacesSelectedAtOnce = true;

        priceDetails.refundablePrice.value = reservationVariant.amount;

        if (reservationVariant.nonRefundableAmount) {
            priceDetails.nonRefundablePrice = {
                value: reservationVariant.nonRefundableAmount,
                currency: priceDetails.price.currency,
            };

            priceDetails.nonRefundablePlaces = sortedCoachPlaces.map(
                ({number}) => number,
            );
        }

        (Object.entries(passengers) as [PASSENGERS_TYPES, number][]).forEach(
            ([passengerType, placesCount]) => {
                times(placesCount).forEach(() => {
                    const isBaby = passengerType === PASSENGERS_TYPES.BABIES;

                    priceDetails.places[passengerType].push({
                        numbers: isBaby ? [] : orderPlaces,
                        withoutPlace: isBaby,
                    });
                });
            },
        );
    } else {
        sortedCoachPlaces.forEach((place, index) => {
            // Рассаживаем детей на самые дорогие места по взрослому тарифу, чтобы минимизировать конечную цену
            const passengerType =
                index < passengers[PASSENGERS_TYPES.CHILDREN]
                    ? PASSENGERS_TYPES.CHILDREN
                    : PASSENGERS_TYPES.ADULTS;

            const placeTariff = getSuitablePriceByCoach(
                passengerType,
                coach,
                place.adultTariff,
                place.childTariff,
            );
            const nonRefundablePlaceTariff = getSuitablePriceByCoach(
                passengerType,
                coach,
                place.adultNonRefundableTariff,
                place.childNonRefundableTariff,
            );

            const currentPlacePrice =
                nonRefundableTariff && nonRefundablePlaceTariff
                    ? nonRefundablePlaceTariff
                    : placeTariff;

            priceDetails.places[passengerType].push({
                numbers: [place.number],
                price: currentPlacePrice,
                withoutPlace: false,
            });
            priceDetails.refundablePrice.value += placeTariff.value;

            if (nonRefundablePlaceTariff) {
                priceDetails.nonRefundablePlaces.push(place.number);
            }

            if (priceDetails.nonRefundablePrice) {
                priceDetails.nonRefundablePrice.value +=
                    nonRefundablePlaceTariff
                        ? nonRefundablePlaceTariff.value
                        : placeTariff.value;
            } else {
                priceDetails.nonRefundablePrice = nonRefundablePlaceTariff
                    ? clone(nonRefundablePlaceTariff)
                    : clone(placeTariff);
            }
        });

        // Дополняем массив детализации детьми
        // Отображаем кол-во детей не превышающее количество взрослых с местом
        const babiesCount = Math.min(
            passengers[PASSENGERS_TYPES.BABIES],
            priceDetails.places[PASSENGERS_TYPES.ADULTS].length,
        );

        for (let i = 0; i < babiesCount; i++) {
            priceDetails.places[PASSENGERS_TYPES.BABIES].push({
                numbers: [],
                price: {
                    value: 0,
                    currency: baseCurrency,
                },
                withoutPlace: true,
            });
        }
    }

    priceDetails.nonRefundablePlaces.sort((a, b) => a - b);

    if (nonRefundableTariff && priceDetails.nonRefundablePrice) {
        priceDetails.price = priceDetails.nonRefundablePrice;
    } else {
        priceDetails.price = priceDetails.refundablePrice;
    }

    return extendTrainPriceDetailsByBedding(priceDetails, {
        coach,
        orderPlaces,
        beddingOption,
    });
}
