import {PASSENGERS_TYPES} from 'projects/trains/constants/passengersTypes';

import {
    IAdditionalSchemeInfo,
    ITrainsCoach,
    TrainsPassengersCount,
} from 'reducers/trains/order/types';

import getBeddingTariff from 'projects/trains/lib/order/getBeddingTariff';
import {countPassengersWithPlaces} from 'projects/trains/lib/order/passengers/utils';
import getMinPlacesPricesGroupedByAgeGroups from 'projects/trains/lib/order/price/getMinPlacesPricesGroupedByAgeGroups';
import checkSchemeHasGroupedPlaces from 'projects/trains/lib/order/coaches/checkSchemeHasGroupedPlaces';
import getSuitableReservationVariantPrice from 'projects/trains/lib/order/price/getSuitableReservationVariantPrice';

interface IGetMinOrderPriceParams {
    /**
     * Выбранное количество пассажиров
     */
    passengers: TrainsPassengersCount;

    /**
     * Выбранный вагон
     */
    coach: ITrainsCoach | null;

    /**
     * Дополнительная информация о местах в вагоне
     * Используется для покупки групп мест
     */
    additionalSchemeInfo: IAdditionalSchemeInfo[];
}

/**
 * Функция возвращает минимально возможную стоимость заказа
 *
 * @param order - заказ
 */
export default function getMinOrderPrice({
    passengers,
    coach,
    additionalSchemeInfo,
}: IGetMinOrderPriceParams): number {
    if (!coach) {
        return 0;
    }

    const pricesByType = getMinPlacesPricesGroupedByAgeGroups({
        places: coach.places,
        priceWithoutPlaces: coach.priceWithoutPlaces,
    });
    const passengersWithPlaces = countPassengersWithPlaces(passengers);

    if (checkSchemeHasGroupedPlaces(coach, additionalSchemeInfo)) {
        return getSuitableReservationVariantPrice(coach, passengers);
    }

    const price = (
        Object.entries(passengers) as [PASSENGERS_TYPES, number][]
    ).reduce((totalPrice, [passengersType, passengersCount]) => {
        if (passengersType === PASSENGERS_TYPES.BABIES) {
            return totalPrice;
        }

        return totalPrice + passengersCount * pricesByType[passengersType];
    }, 0);

    return price + getBeddingTariff(coach) * passengersWithPlaces;
}
