import clone from 'lodash/clone';

import {BASE_CURRENCY} from 'projects/trains/lib/order/getTrainPriceDetails/constants';

import {
    IAdditionalSchemeInfo,
    ITrainsCoach,
    TrainsPassengersCount,
} from 'reducers/trains/order/types';
import {ITrainPriceDetails} from 'projects/trains/lib/order/getTrainPriceDetails/types/ITrainPriceDetails';
import {TrainBookedTariffCode} from 'server/api/TrainsBookingApi/types/ITrainsOrderInfoTicket';
import {ITrainGenericService} from 'server/api/GenericOrderApi/types/common/TGenericService';
import ITrainTicket from 'server/api/GenericOrderApi/types/common/service/ITrainServiceInfo/ITrainPassenger/ITrainTicket';

import {getTariffInfoCodes} from 'projects/trains/lib/order/tariffsInfo';
import getInitialTrainPriceDetails from 'projects/trains/lib/order/getTrainPriceDetails/getInitialTrainPriceDetails';
import {getTicketPlaces} from 'projects/trains/lib/order/places';
import extendTrainPriceDetailsByInsurance from 'projects/trains/lib/order/getTrainPriceDetails/extendTrainPriceDetailsByInsurance';
import getTrainBookedTariffText from 'projects/trains/lib/order/getTrainBookedTariffText';

import * as detailsKeyset from 'i18n/trains-order-summary-details';

interface IGetPreciseTrainPriceDetailsParams {
    trainService: ITrainGenericService;
    passengers: TrainsPassengersCount;
    coach: ITrainsCoach | null;
    additionalSchemeInfo: IAdditionalSchemeInfo[];
    insuranceIncluded: boolean;
}

/**
 * Функция возвращает точную цену на выбранные места.
 * Считаем после резервирования заказа
 */
export default function getPreciseTrainPriceDetails(
    params: IGetPreciseTrainPriceDetailsParams,
): ITrainPriceDetails {
    const {
        trainService,
        passengers,
        coach,
        additionalSchemeInfo,
        insuranceIncluded,
    } = params;

    const tariffs = getTariffInfoCodes(coach?.tariffsInfo ?? []);
    const result = getInitialTrainPriceDetails({
        tariffs,
        coach,
        passengers,
        additionalSchemeInfo,
        isApproximatePrice: false,
    });

    trainService.trainInfo.passengers.forEach(
        ({ticket, nonRefundableTariff}) => {
            if (!ticket) {
                return;
            }

            const {places, amount} = ticket;

            result.price.value += amount.value;

            const title = getTitle(ticket);
            const code = getCode(ticket);
            const withoutPlace = code === TrainBookedTariffCode.BABY;

            if (!result.places[code]) {
                result.places[code] = [];
            }

            const numbers = getTicketPlaces(places)
                .map(Number)
                .filter(Number.isFinite);

            result.places[code].push({
                title,
                withoutPlace,
                numbers,
                price: {
                    value: amount.value,
                    currency: BASE_CURRENCY,
                },
            });

            if (nonRefundableTariff) {
                result.nonRefundablePlaces.push(...numbers);
            }
        },
    );

    const hasNonRefundableTariff = trainService.trainInfo.passengers.some(
        p => p.nonRefundableTariff,
    );

    if (hasNonRefundableTariff) {
        result.nonRefundablePrice = clone(result.price);
    }

    return extendTrainPriceDetailsByInsurance(result, {
        insuranceIncluded,
        trainService,
    });
}

function getTitle(ticket: ITrainTicket): string {
    if (ticket.bookedTariffCode) {
        return getTrainBookedTariffText(ticket.bookedTariffCode);
    }

    return detailsKeyset.rawDashTariffDashTitle({
        tariff: ticket.rawTariffTitle.toLowerCase(),
    });
}

function getCode(ticket: ITrainTicket): string {
    const {tariffInfo, rawTariffTitle, bookedTariffCode} = ticket;

    return (
        bookedTariffCode || (tariffInfo && tariffInfo.code) || rawTariffTitle
    );
}
