import {TrainsInsuranceStatus} from 'server/api/TrainsBookingApi/types/TrainsInsuranceStatus';
import {ITrainGenericService} from 'server/api/GenericOrderApi/types/common/TGenericService';
import {isNotNull, isNotUndefined} from 'types/utilities';
import {IGenericOrderInfo} from 'server/api/GenericOrderApi/types/common/IGenericOrderInfo';

import IPrice from 'utilities/currency/PriceInterface';
import sumPrice from 'utilities/price/sumPrice';
import {getTrainServices} from 'projects/trains/lib/order/getTrainServices';

/**
 * Возвращает стоимость страховки для всех пассажиров в одном поезде (услуге)
 * @param trainService - данные поезда в заказе
 */
export function getInsuranceAmountByService(
    trainService: ITrainGenericService | null,
): IPrice | null {
    const {passengers, insuranceStatus} = trainService?.trainInfo || {};

    if (
        !insuranceStatus ||
        [
            TrainsInsuranceStatus.DISABLED,
            TrainsInsuranceStatus.NEW,
            TrainsInsuranceStatus.PRICING_FAILED,
        ].includes(insuranceStatus) ||
        !passengers
    ) {
        return null;
    }

    return sumPrice(
        ...passengers
            .map(passenger => passenger.insurance?.amount)
            .filter(isNotUndefined),
    );
}

/**
 * Сумма всех страховок для каждой услуги
 * @param orderInfo
 */
export function getTotalInsuranceAmountByOrderInfo(
    orderInfo: IGenericOrderInfo | null,
): IPrice | null {
    const insurancePrices = getTrainServices(orderInfo)
        .map(getInsuranceAmountByService)
        .filter(isNotNull);

    if (!insurancePrices.length) {
        return null;
    }

    return sumPrice(...insurancePrices);
}
