import {SAPSAN_TRAIN_NAME} from 'projects/trains/constants/rawTrainNames';
import {TRAIN_COACH_TYPE} from 'projects/trains/constants/coachType';

import {EPlacesView} from 'types/trains/booking/EPlacesView';
import {
    ITrainOrderState,
    TrainsPassengersCount,
} from 'reducers/trains/order/types';

import {params} from 'utilities/metrika';
import {
    getDataForCount,
    getDataForGenderTrigger,
    getDataForStoreyTrigger,
    getDataForRelativeArrangementOfPlaces,
} from 'projects/trains/lib/order/metrika/emptySchemaUsabilityControls';
import {countPassengersWithPlaces} from 'projects/trains/lib/order/passengers/utils';

export const COMPARTMENT_TRIGGER = 'singleCompartmentTrigger';
export const GENDER_TRIGGER = 'gender';
export const PLATZKARTE_PLACEMENT = 'platzkartePlacement';
export const STOREY_TRIGGER = 'storey';
export const UPPER_BOTTOM_COUNT = 'upperBottomCount';
export const ADDITIONAL_REQUIREMENTS_TRIGGER = 'additionalRequirements';

/**
 * Функция возвращает два отсортированных по алфавиту массива:
 * - availableRequirements содержит перечисление всех требований к местам,
 *   которые были доступны пользователю при выборе места в условиях отсутствия схемы вагона
 * - usedRequirements содержит перечисление всех указанных требований к местам,
 *   которые были доступны пользователю при выборе места в условиях отсутствия схемы вагона
 */
function getAvailableAndUsedRequirements(
    train: ITrainOrderState,
    passengers: TrainsPassengersCount,
): {
    availableRequirements: string[];
    usedRequirements: string[];
} | null {
    const trainDetails = train?.trainDetails?.trainDetails;
    const coach = train?.coach;
    const gender = train?.gender;
    const requirements = train?.requirements;

    if (!coach || !trainDetails || !requirements) {
        return null;
    }

    const {type: coachType, twoStorey} = coach;
    const rawTrainName = trainDetails?.rawTrainName;

    const countPassengers = countPassengersWithPlaces(passengers);

    const requirementsUsageInfo: Record<string, [boolean, boolean]> = {
        [COMPARTMENT_TRIGGER]: getDataForRelativeArrangementOfPlaces(
            {coachType, requirements},
            TRAIN_COACH_TYPE.COMPARTMENT,
            countPassengers > 1,
        ),
        [GENDER_TRIGGER]: getDataForGenderTrigger(coach, gender),
        [PLATZKARTE_PLACEMENT]: getDataForRelativeArrangementOfPlaces(
            {coachType, requirements},
            TRAIN_COACH_TYPE.PLATZKARTE,
            true,
        ),
        [STOREY_TRIGGER]: getDataForStoreyTrigger(twoStorey, requirements),
        [UPPER_BOTTOM_COUNT]: getDataForCount({coachType, requirements}, [
            TRAIN_COACH_TYPE.PLATZKARTE,
            TRAIN_COACH_TYPE.COMPARTMENT,
        ]),
        [ADDITIONAL_REQUIREMENTS_TRIGGER]: [
            rawTrainName === SAPSAN_TRAIN_NAME,
            Boolean(requirements.additional),
        ],
    };

    const availableRequirements = Object.keys(requirementsUsageInfo)
        .filter(key => requirementsUsageInfo[key][0])
        .sort();

    const usedRequirements = availableRequirements
        .filter(key => requirementsUsageInfo[key][1])
        .sort();

    return {availableRequirements, usedRequirements};
}

/**
 * Если пользователь использовал требования выбора мест - отправляем их параметры в метрику
 */
export default function reachGoalUserRequirements(
    train: ITrainOrderState,
    passengers: TrainsPassengersCount,
): void {
    const placesViewType = train.placesViewType;

    if (!placesViewType || placesViewType === EPlacesView.SCHEMAS) {
        return;
    }

    const availableAndUsedRequirements = getAvailableAndUsedRequirements(
        train,
        passengers,
    );

    if (!availableAndUsedRequirements) {
        return;
    }

    const {availableRequirements, usedRequirements} =
        availableAndUsedRequirements;

    params({
        trains: {
            availableRequirements: {
                [availableRequirements.join('-')]: usedRequirements.join('-'),
            },
        },
    });
}
