import invert from 'lodash/invert';

import {PASSENGERS_TYPES} from 'projects/trains/constants/passengersTypes';
import {TRAINS_DOCUMENT_TYPES} from 'projects/trains/constants/documentTypes';
import {EDocumentType} from 'constants/document/documentTypes';

/**
 * Значения для записной книжки путешественника
 */
export const DOCUMENTS_TRAVELERS_KEYS: Record<
    TRAINS_DOCUMENT_TYPES,
    EDocumentType
> = {
    [TRAINS_DOCUMENT_TYPES.RUSSIAN_PASSPORT]:
        EDocumentType.RU_NATIONAL_PASSPORT,
    [TRAINS_DOCUMENT_TYPES.BIRTH_CERTIFICATE]:
        EDocumentType.RU_BIRTH_CERTIFICATE,
    [TRAINS_DOCUMENT_TYPES.RUSSIAN_INTERNATIONAL_PASSPORT]:
        EDocumentType.RU_FOREIGN_PASSPORT,
    [TRAINS_DOCUMENT_TYPES.FOREIGN_DOCUMENT]: EDocumentType.OTHER,
    [TRAINS_DOCUMENT_TYPES.SAILOR_PASSPORT]: EDocumentType.RU_SEAMAN_PASSPORT,
    [TRAINS_DOCUMENT_TYPES.MILITARY_CARD]: EDocumentType.RU_MILITARY_ID,
};

/**
 * Обратная DOCUMENTS_TRAVELERS_KEYS карта соответствия
 */
export const DOCUMENT_TYPE_BY_TRAVELER_DOCUMENT_TYPE = invert(
    DOCUMENTS_TRAVELERS_KEYS,
);

export const ALL_DOCUMENT_TYPES = [
    TRAINS_DOCUMENT_TYPES.RUSSIAN_PASSPORT,
    TRAINS_DOCUMENT_TYPES.BIRTH_CERTIFICATE,
    TRAINS_DOCUMENT_TYPES.RUSSIAN_INTERNATIONAL_PASSPORT,
    TRAINS_DOCUMENT_TYPES.FOREIGN_DOCUMENT,
    TRAINS_DOCUMENT_TYPES.SAILOR_PASSPORT,
    TRAINS_DOCUMENT_TYPES.MILITARY_CARD,
];

export const CHILDREN_DOCUMENT_TYPES = [
    TRAINS_DOCUMENT_TYPES.BIRTH_CERTIFICATE,
    TRAINS_DOCUMENT_TYPES.RUSSIAN_INTERNATIONAL_PASSPORT,
    TRAINS_DOCUMENT_TYPES.FOREIGN_DOCUMENT,
];

export const DOCUMENT_TYPES_BY_AGE = {
    [PASSENGERS_TYPES.ADULTS]: ALL_DOCUMENT_TYPES,
    [PASSENGERS_TYPES.CHILDREN]: CHILDREN_DOCUMENT_TYPES,
    [PASSENGERS_TYPES.BABIES]: CHILDREN_DOCUMENT_TYPES,
};

/**
 * Возвращает список всех доступных типов документов для заданной возрастной группы
 * @param ageGroup - возрастная группа пассажира
 * @param types - список допустимых типов документов для данного поезда
 */
export const getAllowedDocumentTypes = (
    ageGroup: PASSENGERS_TYPES,
    types: TRAINS_DOCUMENT_TYPES[],
): TRAINS_DOCUMENT_TYPES[] => {
    const ageDocuments = DOCUMENT_TYPES_BY_AGE[ageGroup] ?? [];

    return ageDocuments.filter(type => types.includes(type));
};

/**
 * Возвращает дефолтный тип документа для заданной возрастной группы
 * @param ageGroup - возрастная группа пассажира
 * @param types - список допустимых типов документов для данного поезда
 */
export const getDefaultDocumentType = (
    ageGroup: PASSENGERS_TYPES,
    types: TRAINS_DOCUMENT_TYPES[],
): TRAINS_DOCUMENT_TYPES => getAllowedDocumentTypes(ageGroup, types)[0];
