import moment from 'moment-timezone';

import {EOrderErrorType} from 'projects/trains/constants/orderErrors';

import {IOrderError} from 'reducers/trains/order/types';
import {ITrainsTariffCategory} from 'server/api/TrainsApi/types/ITrainsDetailsApiResponse';
import EOriginalPaymentErrorCode from 'types/common/EOriginalPaymentErrorCode';
import {IGenericOrderInfo} from 'server/api/GenericOrderApi/types/common/IGenericOrderInfo';

import {getNow} from 'utilities/dateUtils';
import {insertJSXIntoKey} from 'utilities/tanker/insertJSXIntoKey';

import * as i18nPaymentErrorBlock from 'i18n/trains-payment-errors';
import * as i18nOrderPaymentBlock from 'i18n/trains-order-payment';

import SupportPhone from 'components/SupportPhone/SupportPhone';
import TrainsFeedbackLink from 'projects/trains/components/TrainsFeedbackLink/TrainsFeedbackLink';

/**
 * getPaymentTimeout - вернет разницу в милисекундах между текущим временем и более поздним временем из
 * момента протухания заказа и следующей проверки статуса заказа у партнера
 */
export function getPaymentTimeout({expiresAt}: IGenericOrderInfo): number {
    return moment(expiresAt).diff(getNow());
}

/**
 * Функция вернет ошибку оплаты с соответствующим текстом
 * @param {Object} [orderInfo] - данные заказа
 * @param {Object} [tariffCategories] - данные о возрастных категориях
 * @return {{type: EOrderErrorType, title: string, message: string}}
 */
export function getPaymentError(
    orderInfo: IGenericOrderInfo,
    tariffCategories?: ITrainsTariffCategory[],
): IOrderError {
    const paymentCode = orderInfo.payment?.errorInfo || null;

    const message =
        getPaymentErrorText(paymentCode) ||
        insertJSXIntoKey(
            i18nPaymentErrorBlock.unknownDashErrorDashSupportDashPhone,
        )({
            supportPhone: <SupportPhone />,
            feedbackLink: (
                <TrainsFeedbackLink
                    orderInfo={orderInfo}
                    tariffCategories={tariffCategories}
                >
                    {i18nPaymentErrorBlock.unknownDashErrorDashFeebackDashLink()}
                </TrainsFeedbackLink>
            ),
        });

    return {
        type: EOrderErrorType.PAYMENT,
        title: i18nPaymentErrorBlock.errorDashTitle(),
        message,
    };
}

/**
 * getWaiterInfo - возвращает информацию о состоянии лоадера при бронировании и оплате
 */
export function getWaiterInfo({
    loaded,
    complete,
    isTrustIframeErrorVisible,
}: {
    loaded: boolean;
    complete: boolean;
    isTrustIframeErrorVisible: boolean;
}): {showWaiter: boolean; waiterText: string} {
    const showWaiter = Boolean(
        !loaded || complete || isTrustIframeErrorVisible,
    );

    if (!showWaiter) {
        return {
            showWaiter,
            waiterText: '',
        };
    }

    if (!complete) {
        return {
            showWaiter,
            waiterText: i18nOrderPaymentBlock.waiterDashReservationDashText(),
        };
    }

    return {
        showWaiter,
        waiterText: i18nOrderPaymentBlock.waiterDashSuccessDashText(),
    };
}

function getPaymentErrorText(
    paymentCode: EOriginalPaymentErrorCode | null,
): string | null {
    switch (paymentCode) {
        case EOriginalPaymentErrorCode.NOT_ENOUGH_FUNDS:
            return i18nPaymentErrorBlock.notEnoughFunds();
        case EOriginalPaymentErrorCode.EXPIRED_CARD:
            return i18nPaymentErrorBlock.expiredCard();
        case EOriginalPaymentErrorCode.LIMIT_EXCEEDED:
            return i18nPaymentErrorBlock.limitExceeded();
        case EOriginalPaymentErrorCode.AUTHORIZATION_REJECT:
            return i18nPaymentErrorBlock.authorizationReject();
        case EOriginalPaymentErrorCode.TRANSACTION_NOT_PERMITTED:
            return i18nPaymentErrorBlock.transactionNotPermitted();
        case EOriginalPaymentErrorCode.RESTRICTED_CARD:
            return i18nPaymentErrorBlock.restrictedCard();
        case EOriginalPaymentErrorCode.BLACKLISTED:
            return i18nPaymentErrorBlock.blacklisted();
        case EOriginalPaymentErrorCode.FAIL_3DS:
            return i18nPaymentErrorBlock.fail3Ds();
        case EOriginalPaymentErrorCode.USER_CANCELLED:
            return i18nPaymentErrorBlock.userCancelled();

        default:
            return null;
    }
}
