import flatMap from 'lodash/flatMap';
import last from 'lodash/last';
import _unionBy from 'lodash/unionBy';

import {ITrainsCoach} from 'reducers/trains/order/types';
import ITrainPassenger from 'server/api/GenericOrderApi/types/common/service/ITrainServiceInfo/ITrainPassenger/ITrainPassenger';
import ITrainPlaceWithType from 'server/api/GenericOrderApi/types/common/service/ITrainServiceInfo/ITrainPassenger/ITrainPlaceWithType';
import ETrainDisplayReservationPlaceType from 'server/api/GenericOrderApi/types/common/service/ITrainServiceInfo/ITrainPassenger/ETrainDisplayReservationPlaceType';

import getMaxPlacesReservationCountByType from 'projects/trains/lib/order/places/getMaxPlacesReservationCountByType';

const VIRTUAL_PLACE = '000';

const isRealPlace = (place: string): boolean => place !== VIRTUAL_PLACE;

/**
 * Вырезает нули у номера который содержит только чиловое значение
 * @param {string} place - номер места
 * @return {string}
 */
export const trimZero = (place: string): string => {
    const placeNum = Number(place);

    return isNaN(placeNum) ? place : String(placeNum);
};

/**
 * Возвращает массив номеров мест из билета, исключая специальные места с номером "000"
 * @param places
 */
export const getTicketPlaces = (places: ITrainPlaceWithType[]): string[] =>
    places
        .map(({number}) => number)
        .filter(isRealPlace)
        .map(trimZero);

/**
 * Проверяет, совпадают ли типы мест и возвращает этот тип
 * @param places
 */
export const getTicketPlacesType = (
    places: ITrainPlaceWithType[],
): string | undefined => {
    const unionPlacesByType = _unionBy(places, place => place.type);

    if (unionPlacesByType.length !== 1) {
        return undefined;
    }

    return unionPlacesByType[0]?.typeText;
};

/**
 * Возвращает массив номеров мест для всех билетов пассажира
 */
const getPassengerPlaces = ({ticket}: ITrainPassenger): string[] =>
    getTicketPlaces(ticket?.places || []);

/**
 * Возвращает массив номеров мест для всех пассажиро
 */
export const getActualPlaces = (passengers: ITrainPassenger[]): string[] =>
    flatMap(passengers, getPassengerPlaces);

/**
 * Возвращает список мест вместе с описанием яруса
 */
export const getPlacesWithDescription = (
    passengers: ITrainPassenger[],
): string[] =>
    flatMap(passengers, ({ticket}) => {
        if (!ticket) {
            return '';
        }

        const lastTicket = last(ticket.places);
        const description =
            lastTicket?.type ===
            ETrainDisplayReservationPlaceType.FOR_PASSENGER_WITH_PET
                ? ''
                : lastTicket?.typeText;
        const places = getTicketPlaces(ticket.places);

        if (!places.length) {
            return '';
        }

        const placesString = places.join(', ');

        // В описании "й" это два символа: "и" + ̆  (https://www.fileformat.info/info/unicode/char/0306/index.htm)
        // поэтому в FF отображается как двойной пробел
        // @see https://st.yandex-team.ru/TRAINS-1434
        const formattedDescription = description?.replace(/и\u0306/g, 'й');

        function addDescription(prefix: string): string {
            return [prefix, formattedDescription].filter(Boolean).join(' ');
        }

        return addDescription(placesString);
    }).filter(Boolean);

enum EPlaceStatus {
    SELECTED = 'selected',
    ACTUAL = 'actual',
    BOTH = 'both',
}

export function diffPlaces(
    selectedPlaces: number[],
    actualPlaces: string[],
): Record<EPlaceStatus, string[]> {
    const map: Record<string, EPlaceStatus> = {};

    selectedPlaces.forEach(place => {
        map[String(place)] = EPlaceStatus.SELECTED;
    });

    actualPlaces.forEach(place => {
        map[place] = map[place] ? EPlaceStatus.BOTH : EPlaceStatus.ACTUAL;
    });

    const result: Record<EPlaceStatus, string[]> = {
        [EPlaceStatus.SELECTED]: [],
        [EPlaceStatus.ACTUAL]: [],
        [EPlaceStatus.BOTH]: [],
    };

    Object.keys(map)
        .sort((a, b) => (a > b ? -1 : 1))
        .forEach(place => {
            const resultSection = result[map[place]];
            const insertIndex = resultSection.findIndex(item => item > place);

            resultSection.splice(insertIndex, 0, place);
        });

    return result;
}

export function havePlacesChanged({
    selected,
    actual,
}: Record<EPlaceStatus, string[]>): boolean {
    /**
     * Исключаем случаи, когда мест выбрано меньше чем пассажиров
     */
    return selected.length > 0 && selected.length === actual.length;
}

/**
 * Возвращает максимальное количество мест, которое можно купить в данном вагоне.
 */
export function getMaxPlacesReservationCountByCoach(
    coach: ITrainsCoach,
): number {
    return getMaxPlacesReservationCountByType(coach.placeReservationType);
}
