import _last from 'lodash/last';

import {humanList} from 'utilities/strings/humanList';
import {CHAR_DASH, CHAR_NBSP} from 'utilities/strings/charCodes';

import * as i18nBlock from 'i18n/trainsNonRefundableTariffSelector';

export const convertPlacesToString = ({
    places = [],
    isLowerCase,
    placesCount,
}: {
    places?: (string | number)[];
    isLowerCase?: boolean;
    placesCount?: number;
}): string => {
    const humanPlaceList = humanList(places);
    const placesString = i18nBlock.nonRefundablePlaces({
        places: humanPlaceList ?? '',
        count: placesCount ?? places?.length ?? 1,
    });

    if (isLowerCase) {
        return placesString.toLowerCase();
    }

    return placesString;
};

interface IPlaceRangeNumber {
    startPlaceRangeNumber: number;
    endPlaceRangeNumber?: number;
}

export const prepareAndConvertPlacesRangeToString = ({
    places,
    isLowerCase,
}: {
    places: number[];
    isLowerCase?: boolean;
}): string => {
    if (places.length <= 1) {
        return convertPlacesToString({places, isLowerCase});
    }

    const placeRangeNumbers = [...places]
        .sort(
            (firstPlace, secondPlace) =>
                Number(firstPlace) - Number(secondPlace),
        )
        .reduce<IPlaceRangeNumber[]>((resultPlaceRangeNumbers, place) => {
            const lastPlaceRangeNumber = _last(resultPlaceRangeNumbers);

            if (!lastPlaceRangeNumber) {
                resultPlaceRangeNumbers.push({
                    startPlaceRangeNumber: place,
                });

                return resultPlaceRangeNumbers;
            }

            const {startPlaceRangeNumber, endPlaceRangeNumber} =
                lastPlaceRangeNumber;

            if (place - startPlaceRangeNumber === 0) {
                return resultPlaceRangeNumbers;
            }

            const canAddEndRangeNumber = place - startPlaceRangeNumber === 1;
            const canUpdateEndRangeNumber =
                endPlaceRangeNumber && place - endPlaceRangeNumber === 1;

            if (canAddEndRangeNumber || canUpdateEndRangeNumber) {
                lastPlaceRangeNumber.endPlaceRangeNumber = place;

                return resultPlaceRangeNumbers;
            }

            resultPlaceRangeNumbers.push({
                startPlaceRangeNumber: place,
            });

            return resultPlaceRangeNumbers;
        }, [])
        .map(placesRange => {
            const {startPlaceRangeNumber, endPlaceRangeNumber} = placesRange;

            if (endPlaceRangeNumber) {
                return `${startPlaceRangeNumber}${CHAR_NBSP}${CHAR_DASH}${CHAR_NBSP}${endPlaceRangeNumber}`;
            }

            return startPlaceRangeNumber;
        });

    return convertPlacesToString({
        isLowerCase,
        places: placeRangeNumbers,
        placesCount: places.length,
    });
};
