import intersection from 'lodash/intersection';

import {ITrainsCoach} from 'reducers/trains/order/types';
import {
    IPriceDivision,
    IPriceDivisionWithTheme,
} from 'projects/trains/lib/order/priceScale/types';
import {EPricePalette} from 'types/trains/booking/EPricePalette';

import getCoachInfo from 'projects/trains/lib/order/getCoachInfo';
import {makeCacheable} from 'projects/trains/lib/cache';
import countDivision from 'projects/trains/lib/order/priceScale/countDivision';
import colorizeDivision, {
    isValidDivision,
} from 'projects/trains/lib/order/priceScale/colorizeDivision';
import IPrice from 'utilities/currency/PriceInterface';

/**
 * Класс для построения разбиения по ценам и работы с ним.
 * Используется для построения легенды цен и раскраски схем вагонов
 * P.S. признак постельного белья прокидывается для того чтобы легенда
 * содержала цены с учетом использования постельного белья
 */
export default class PriceScale {
    division: (IPriceDivision | IPriceDivisionWithTheme)[];

    /**
     * Вернёт цветовую схему для заданной цены
     *
     * @param tariff - цена
     */
    getTheme: (tariff: IPrice | null) => EPricePalette | null = makeCacheable(
        (tariff: IPrice | null): EPricePalette | null => {
            if (!tariff) {
                return null;
            }

            const price = tariff.value;
            const division = this.division.find(({values}) =>
                values.includes(price),
            );

            if (!division?.theme) {
                return null;
            }

            return division.theme;
        },
    );

    /**
     * @param coaches - массив вагонов, по которым строится разбиение
     * @param beddingOption - признак использования постельного белья
     */
    constructor(coaches: ITrainsCoach[], beddingOption: boolean) {
        this.division = colorizeDivision(countDivision(coaches, beddingOption));
    }

    /**
     * Вернёт разбиение соответствующее ценам в заданном вагоне
     *
     * @param coach - данные вагона
     * @param beddingOption - признак использования постельного белья
     */
    getDivision(
        coach: ITrainsCoach,
        beddingOption: boolean,
    ): (IPriceDivision | IPriceDivisionWithTheme)[] {
        const {prices} = getCoachInfo(coach, beddingOption);
        const uniqPricesValues = [...new Set(prices.map(({value}) => value))];

        return this.division
            .map<IPriceDivision | IPriceDivisionWithTheme>(item => ({
                ...item,
                values: intersection(item.values, uniqPricesValues),
            }))
            .filter(({values}) => values.length);
    }

    /**
     * Вернёт признак того, что к разбиению применилась раскраска
     */
    isValid(): boolean {
        return isValidDivision(this.division);
    }
}
