import {PASSENGERS_TYPES} from 'projects/trains/constants/passengersTypes';
import {TRAINS_DOCUMENT_TYPES} from 'projects/trains/constants/documentTypes';

import {ITrainPassenger} from 'reducers/trains/order/types';
import {ERestrictionType} from 'projects/trains/lib/order/fields/restrictions/types';
import {IStepRestrictionsData} from 'projects/trains/lib/order/stepRestrictions/types/IStepRestrictionsData';

import {
    createRestriction,
    prioritizeRestrictions,
} from 'projects/trains/lib/restrictions';
import {getDefaultDocumentType} from 'projects/trains/lib/order/passengers/documentTypes';
import {
    areAllHashFieldsFilled,
    BIRTH_DATE_KEY,
    FIRST_NAME_KEY,
    getPassengerHash,
    LAST_NAME_KEY,
    PATRONYMIC_KEY,
} from 'projects/trains/lib/order/passengers/getPassengerHash';

type TExcludedFieldsGetter = (docType: TRAINS_DOCUMENT_TYPES) => string[];

const moreThanOneTicketPerPerson = (
    passengersData: ITrainPassenger[],
    allowedDocumentTypes: TRAINS_DOCUMENT_TYPES[],
    excludedFieldsGetter: TExcludedFieldsGetter,
): boolean => {
    const passengersWithDiscount = passengersData
        .map((passenger: ITrainPassenger): string => {
            const {ageGroup, passengerDocument} = passenger;
            const documentType =
                passengerDocument.type.value ||
                getDefaultDocumentType(ageGroup, allowedDocumentTypes);
            const excludeFields = excludedFieldsGetter(documentType);

            return areAllHashFieldsFilled(passenger, excludeFields)
                ? getPassengerHash(passenger, excludeFields)
                : '';
        })
        .filter(Boolean);

    const uniquePassengersWithDiscount = new Set(passengersWithDiscount);

    return Boolean(
        uniquePassengersWithDiscount.size &&
            passengersWithDiscount.length !== uniquePassengersWithDiscount.size,
    );
};

const moreThanOneDiscountTicketPerChild = createRestriction<
    IStepRestrictionsData,
    boolean
>(
    ERestrictionType.MORE_THAN_ONE_DISCOUNT_TICKET_PER_PERSON,
    ({passengersData, allowedDocumentTypes}) => {
        const filteredPassengers = passengersData.filter(
            ({ageGroup}) => ageGroup !== PASSENGERS_TYPES.ADULTS,
        );

        return moreThanOneTicketPerPerson(
            filteredPassengers,
            allowedDocumentTypes,
            documentType =>
                documentType === TRAINS_DOCUMENT_TYPES.BIRTH_CERTIFICATE
                    ? [
                          FIRST_NAME_KEY,
                          LAST_NAME_KEY,
                          PATRONYMIC_KEY,
                          BIRTH_DATE_KEY,
                      ]
                    : [BIRTH_DATE_KEY],
        );
    },
);

const moreThanOneTicketPerPersonForCppk = createRestriction<
    IStepRestrictionsData,
    boolean
>(
    ERestrictionType.MORE_THAN_ONE_TICKET_PER_PERSON,
    ({passengersData, isCppk, allowedDocumentTypes}) => {
        if (isCppk) {
            return moreThanOneTicketPerPerson(
                passengersData,
                allowedDocumentTypes,
                () => [
                    FIRST_NAME_KEY,
                    LAST_NAME_KEY,
                    PATRONYMIC_KEY,
                    BIRTH_DATE_KEY,
                ],
            );
        }

        return false;
    },
);

export default prioritizeRestrictions([
    moreThanOneTicketPerPersonForCppk,
    moreThanOneDiscountTicketPerChild,
]);
