import {PLACE_RESERVATION_TYPE} from 'projects/trains/constants/placeReservationType';
import {PASSENGERS_TYPES} from 'projects/trains/constants/passengersTypes';
import {TRAIN_COACH_TYPE} from 'projects/trains/constants/coachType';
import {ORDER_STEP} from 'projects/trains/constants/orderSteps';

import {EPlacesView} from 'types/trains/booking/EPlacesView';
import {ERestrictionType} from 'projects/trains/lib/order/fields/restrictions/types';
import {IStepRestrictionsData} from 'projects/trains/lib/order/stepRestrictions/types/IStepRestrictionsData';

import {
    createRestriction,
    prioritizeRestrictions,
} from 'projects/trains/lib/restrictions';
import {getCoachPlacesCount} from 'projects/trains/lib/order/utils';
import areRequirementsAvailableForCoach from 'projects/trains/lib/order/areRequirementsAvailableForCoach';
import isGenderCoach from 'projects/trains/lib/order/gender/isGenderCoach';
import getAvailableGenders from 'projects/trains/lib/order/gender/getAvailableGenders';
import {countPassengersWithPlaces} from 'projects/trains/lib/order/passengers/utils';
import isGenderAvailableForSelect from 'projects/trains/lib/order/gender/isGenderAvailableForSelect';
import areCoachesAutoSeat from 'projects/trains/lib/order/coaches/areCoachesAutoSeat';
import {getMaxPlacesReservationCountByCoach} from 'projects/trains/lib/order/places';
import checkSchemeHasGroupedPlaces from 'projects/trains/lib/order/coaches/checkSchemeHasGroupedPlaces';

const serviceClassNotSelected = createRestriction<
    IStepRestrictionsData,
    boolean
>(ERestrictionType.SERVICE_CLASS_NOT_SELECTED, ({expandedServiceClass}) => {
    return !expandedServiceClass;
});

const genderNotSelected = createRestriction<IStepRestrictionsData, boolean>(
    ERestrictionType.GENDER_NOT_SELECTED,
    ({passengers, coach, gender}) => {
        return Boolean(
            coach &&
                isGenderCoach(coach) &&
                isGenderAvailableForSelect(coach, passengers) &&
                !gender,
        );
    },
);

const notEnoughPlaces = createRestriction<IStepRestrictionsData, boolean>(
    ERestrictionType.NOT_ENOUGH_PLACES,
    ({passengers, coach}) => {
        return Boolean(
            coach &&
                !areCoachesAutoSeat([coach]) &&
                countPassengersWithPlaces(passengers) >
                    getCoachPlacesCount(coach),
        );
    },
);

const notEnoughPlacesInClass = createRestriction<
    IStepRestrictionsData,
    boolean
>(
    ERestrictionType.NOT_ENOUGH_PLACES_IN_CLASS,
    // если для самого подходящего вагона нет мест, следовательно и для всего класса не будет мест
    ({passengers, coach}) => {
        return Boolean(
            coach &&
                areCoachesAutoSeat([coach]) &&
                countPassengersWithPlaces(passengers) >
                    getCoachPlacesCount(coach),
        );
    },
);

const notEnoughGenderPlaces = createRestriction<IStepRestrictionsData, boolean>(
    ERestrictionType.NOT_ENOUGH_GENDER_PLACES,
    ({passengers, coach, gender}) => {
        return Boolean(
            coach &&
                gender &&
                !getAvailableGenders(coach, passengers).includes(gender),
        );
    },
);

const noPassengersInOrder = createRestriction<IStepRestrictionsData, boolean>(
    ERestrictionType.NO_PASSENGERS_IN_ORDER,
    ({passengers}) => {
        return !countPassengersWithPlaces(passengers);
    },
);

const passengersAndPlaces = ({
    passengers,
    coach,
    orderPlaces,
    placesViewType,
    coachType,
    additionalSchemeInfo,
}: IStepRestrictionsData): {
    type: ERestrictionType;
    payload:
        | {
              count?: number;
              max?: number;
          }
        | undefined;
}[] => {
    if (
        (coach &&
            coach.schemaId === null &&
            coach.placeReservationType === PLACE_RESERVATION_TYPE.USUAL) ||
        (placesViewType === EPlacesView.REQUIREMENTS &&
            areRequirementsAvailableForCoach({coach}))
    ) {
        return [];
    }

    const placesCount = orderPlaces.length;
    const passengersCount = countPassengersWithPlaces(passengers);

    if (coach && checkSchemeHasGroupedPlaces(coach, additionalSchemeInfo)) {
        const maxPlacesReservationCount =
            getMaxPlacesReservationCountByCoach(coach);

        if (passengersCount > maxPlacesReservationCount) {
            return [
                {
                    type: ERestrictionType.MAX_PASSENGERS_REACHED,
                    payload: {
                        max: maxPlacesReservationCount,
                        count: passengersCount - maxPlacesReservationCount,
                    },
                },
            ];
        }

        // Для двуместных купе в Стрижах
        if (
            coachType === TRAIN_COACH_TYPE.SUITE &&
            passengers[PASSENGERS_TYPES.ADULTS] === 0 &&
            passengers[PASSENGERS_TYPES.CHILDREN] === 1
        ) {
            return [
                {
                    type: ERestrictionType.ONLY_ONE_CHILDREN_IN_TWO_SEATER_COMPARTMENT,
                    payload: undefined,
                },
            ];
        }

        // Для купе-переговорной в Сапсане
        if (
            coachType === TRAIN_COACH_TYPE.SITTING &&
            passengers[PASSENGERS_TYPES.ADULTS] === 0
        ) {
            return [
                {
                    type: ERestrictionType.ONLY_CHILDREN_IN_MEETING_ROOM_COMPARTMENT,
                    payload: undefined,
                },
            ];
        }

        if (coach.schemaId !== null && placesCount === 0) {
            return [
                {
                    type: ERestrictionType.MORE_PASSENGERS_THAN_PLACES,
                    payload: {
                        count: maxPlacesReservationCount,
                    },
                },
            ];
        }

        return [];
    }

    if (passengersCount > placesCount) {
        return [
            {
                type: ERestrictionType.MORE_PASSENGERS_THAN_PLACES,
                payload: {
                    count: passengersCount - placesCount,
                },
            },
        ];
    }

    if (passengersCount < placesCount) {
        return [
            {
                type: ERestrictionType.LESS_PASSENGERS_THAN_PLACES,
                payload: {
                    count: placesCount - passengersCount,
                },
            },
        ];
    }

    return [];
};

const hasDifferentGenderInTrains = createRestriction<
    IStepRestrictionsData,
    boolean
>(ERestrictionType.HAS_DIFFERENT_GENDER_IN_TRAINS, data => {
    if (data.nextStep.step !== ORDER_STEP.PASSENGERS) {
        return false;
    }

    return data.hasDifferentGenderInTrains;
});

export default prioritizeRestrictions<
    IStepRestrictionsData,
    | boolean
    | {
          count?: number;
          max?: number;
      }
    | undefined
>([
    serviceClassNotSelected,
    notEnoughPlaces,
    notEnoughPlacesInClass,
    noPassengersInOrder,
    passengersAndPlaces,
    genderNotSelected,
    notEnoughGenderPlaces,
    hasDifferentGenderInTrains,
]);
