import {Action} from 'typesafe-actions';

import {GENDER_TYPE} from 'projects/trains/constants/genders';

import {ERestrictionType} from 'projects/trains/lib/order/fields/restrictions/types';
import {isNotNull} from 'types/utilities';
import {IStepRestrictionsData} from 'projects/trains/lib/order/stepRestrictions/types/IStepRestrictionsData';

import {setPlaces} from 'reducers/trains/order/actions/trains';

import {countPassengersWithPlaces} from 'projects/trains/lib/order/passengers/utils';
import {getMaxPlacesReservationCountByCoach} from 'projects/trains/lib/order/places';
import checkSchemeHasGroupedPlaces from 'projects/trains/lib/order/coaches/checkSchemeHasGroupedPlaces';
import {IStepRestriction} from 'projects/trains/lib/order/stepRestrictions/restrictions';

const solutions: PartialRecord<
    ERestrictionType,
    (
        payload: IStepRestriction['payload'],
        data: IStepRestrictionsData,
    ) => null | (() => Action)
> = {
    [ERestrictionType.MORE_PASSENGERS_THAN_PLACES]: (
        payload,
        {
            coach,
            expandedServiceClass,
            additionalSchemeInfo,
            passengers,
            direction,
            index,
        },
    ) => {
        if (!coach || !coach.places) {
            return null;
        }

        const passengersCount = countPassengersWithPlaces(passengers);
        const {places} = coach;

        // Если среди мест есть хотя бы одно гендерное - мы не можем выбирать за пользователя
        if (places.some(({gender}) => gender && gender !== GENDER_TYPE.MIXED)) {
            return null;
        }

        if (
            expandedServiceClass &&
            ((checkSchemeHasGroupedPlaces(coach, additionalSchemeInfo) &&
                passengersCount <= places.length &&
                places.length <= getMaxPlacesReservationCountByCoach(coach)) ||
                passengersCount === places.length)
        ) {
            return (): Action =>
                setPlaces({
                    direction,
                    index,
                    data: places.map(({number}) => number),
                });
        }

        return null;
    },
};

/**
 * Вернёт массив функций, позволяющих разрешить возникшие ограничения
 *
 * @param restrictions - массив ограничений
 * @param data - вспомогательные данные
 */
export const getSolutions = (
    restrictions: IStepRestriction[],
    data: IStepRestrictionsData,
): (() => Action)[] =>
    restrictions
        .map(({type, payload}) => {
            const solution = solutions[type];

            return solution ? solution(payload, data) : null;
        })
        .filter(isNotNull);
