import forEach from 'lodash/forEach';

import {ITrainsCoach} from 'reducers/trains/order/types';
import {TTrainsStoreOrderSegment} from 'projects/trains/lib/segments/types';

import getExtremes from 'projects/trains/lib/getExtremes';
import getCoachTypesData from 'projects/trains/lib/order/tariffsConsistency/getCoachTypesData';

/**
 * Возвращает расхождение по тарифам актуальных данных и того, что берется из сегмента
 * @param coaches - вагоны из актуальных данных
 * @param segment
 */
export default function getTariffsInconsistency(
    coaches: ITrainsCoach[],
    segment: TTrainsStoreOrderSegment,
): {
    placeDifference: number;
    minPriceDifference: number;
} {
    const inconsistency = {
        placeDifference: 0,
        minPriceDifference: 0,
    };

    const tariffs = 'tariffs' in segment ? segment.tariffs : null;

    if (!tariffs || !tariffs.classes) {
        return inconsistency;
    }

    const orderCoachesData = getCoachTypesData(coaches);

    let minOrderPrice: number | null = null;
    let minSegmentPrice: number | null = null;

    forEach(tariffs.classes, (segmentCoachData, coachType) => {
        if (!segmentCoachData) {
            return;
        }

        // На выдаче могут быть протухшие типы вагонов
        const orderCoachData = orderCoachesData[coachType];

        const orderPrice =
            orderCoachData && getExtremes(orderCoachData.prices).min;

        const placeDifference = orderCoachData
            ? segmentCoachData.seats - orderCoachData.count
            : segmentCoachData.seats;

        if (
            orderPrice &&
            (!minOrderPrice || orderPrice.value < minOrderPrice)
        ) {
            minOrderPrice = orderPrice.value;
        }

        if (
            !minSegmentPrice ||
            segmentCoachData.price.value < minSegmentPrice
        ) {
            minSegmentPrice = segmentCoachData.price.value;
        }

        inconsistency.placeDifference += placeDifference;
    });

    if (minOrderPrice && minSegmentPrice) {
        inconsistency.minPriceDifference = minOrderPrice - minSegmentPrice;
    }

    return inconsistency;
}
