import {PASSENGERS_TYPES} from 'projects/trains/constants/passengersTypes';

import {ITrainsCoachTariff, ITrainPassenger} from 'reducers/trains/order/types';
import {TrainBookedTariffCode} from 'server/api/TrainsBookingApi/types/ITrainsOrderInfoTicket';

import {getPassengerHash} from 'projects/trains/lib/order/passengers/getPassengerHash';
import {getBoardingAge} from 'utilities/dateUtils/getBoardingAge';

/**
 * Возвращает коды тарифов.
 * Меняем порядок: от общего к конкретному.
 *
 * @param tariffsInfo - тарифы.
 *
 * @returns коды тарифов.
 */
export function getTariffInfoCodes(
    tariffsInfo: ITrainsCoachTariff[],
): TrainBookedTariffCode[] {
    return tariffsInfo.map(({code}) => code).reverse();
}

/**
 * Поиск подходящего тарифа для пассажира.
 *
 * Если withDocument=false, то наличие необходимого документа у пассажира не проверяется.
 */
function getTariff({
    passenger,
    age,
    tariffs,
    withDocument = true,
}: {
    passenger: ITrainPassenger;
    age: number;
    tariffs: ITrainsCoachTariff[];
    withDocument: boolean;
}): ITrainsCoachTariff {
    return (
        tariffs.find(
            ({
                minAge,
                minAgeIncludesBirthday,
                maxAge,
                maxAgeIncludesBirthday,
                withoutPlace,
                needDocument,
            }) =>
                (minAge < age || (minAgeIncludesBirthday && minAge === age)) &&
                (maxAge > age || (maxAgeIncludesBirthday && maxAge === age)) &&
                (passenger.ageGroup === PASSENGERS_TYPES.BABIES) ===
                    withoutPlace &&
                (withDocument
                    ? needDocument === passenger.hasDiscountDocument
                    : true),
        ) || tariffs[tariffs.length - 1]
    );
}

/**
 * Распределение тарифов по пассажирам.
 *
 * Если withDocument=false, то наличие необходимого документа у пассажира не проверяется.
 *
 * @param passengers - Пассажиры.
 * @param segment - Сегмент.
 * @param tariffs - Доступные тарифы для вагона.
 * @param [withDocument=true] - проверка необходимости документа для тарифа.
 *
 * @returns Тарифы для каждого пассажира.
 */
export function getPassengersTariffs({
    passengers,
    departure,
    tariffs,
    withDocument = true,
}: {
    passengers: ITrainPassenger[];
    departure: string | undefined;
    tariffs: ITrainsCoachTariff[];
    withDocument?: boolean;
}): ITrainsCoachTariff[] {
    const passengersHashes = passengers.map(passenger =>
        getPassengerHash(passenger),
    );

    return passengers.map((passenger, index) => {
        const firstIndexForSamePassenger = passengersHashes.findIndex(
            hash => hash === passengersHashes[index],
        );

        if (firstIndexForSamePassenger < index) {
            return tariffs[tariffs.length - 1];
        }

        const birthDate = passenger.birthDate.value;

        if (!birthDate || !departure) {
            return tariffs[tariffs.length - 1];
        }

        const age = getBoardingAge(birthDate, departure);

        return getTariff({passenger, age, tariffs, withDocument});
    });
}
