import {TRAIN_BONUS_CARDS} from 'projects/trains/constants/bonusCards';
import {PASSENGERS_TYPES} from 'projects/trains/constants/passengersTypes';
import {GENDER_TYPE} from 'projects/trains/constants/genders';

import {
    IStoreTrainTravelerPassenger,
    IStoreTravelerPassengerDocument,
    ITrainPassenger,
    PassengerBonusCardsType,
} from 'reducers/trains/order/types';
import {ITrainsTariffCategory} from 'server/api/TrainsApi/types/ITrainsDetailsApiResponse';
import {ITrainsLoyaltyCard} from 'server/api/TrainsApi/types/ITrainsLoyaltyCard';
import {EGender} from 'types/common/document/EGender';

import {getAgeGroup} from 'projects/trains/lib/order/fields/birthDate';
import {getBoardingAge} from 'utilities/dateUtils/getBoardingAge';

/**
 * Функция преобразует пассажира из записной книжки в пассажира со структурой как в order.passengersData
 *
 * Нужно, например, для дальнейшей валидации функцией getValidatedAndUpdatedPassenger
 *
 * @param params
 * @param params.travelerPassenger — пассажир из записной книжки
 * @param params.travelerPassengerDocument — документ пассажира из записной книжки
 * @param params.lastSegmentDeparture — время отправления последнего сегмента
 * @param params.tariffCategories — возрастные категории
 * @param params.allowedLoyaltyCards — доступные для ввода карты лояльности
 */
export default function getStoreLikePassengerFromTravelerPassenger({
    travelerPassenger,
    travelerPassengerDocument,
    lastSegmentDeparture,
    tariffCategories,
    allowedLoyaltyCards,
}: {
    travelerPassenger: IStoreTrainTravelerPassenger;
    travelerPassengerDocument: IStoreTravelerPassengerDocument;
    lastSegmentDeparture: string | undefined;
    tariffCategories: ITrainsTariffCategory[];
    allowedLoyaltyCards: TRAIN_BONUS_CARDS[];
}): ITrainPassenger {
    return {
        firstName: travelerPassengerDocument.firstName,
        lastName: travelerPassengerDocument.lastName,
        patronymic: travelerPassengerDocument.patronymic,
        birthDate: travelerPassenger.birthDate,
        ageGroup:
            travelerPassenger.birthDate?.value && lastSegmentDeparture
                ? getAgeGroup(
                      getBoardingAge(
                          travelerPassenger.birthDate.value,
                          lastSegmentDeparture,
                      ),
                      tariffCategories,
                  ) ?? PASSENGERS_TYPES.ADULTS
                : PASSENGERS_TYPES.ADULTS,
        passengerDocument: {
            type: travelerPassengerDocument.type,
            number: travelerPassengerDocument.number,
            country: travelerPassengerDocument.country,
            validDate: travelerPassengerDocument.validDate,
        },
        emailOrPhone: {
            value: (travelerPassenger.email || travelerPassenger.phone) ?? '',
        },
        bonusCards: (travelerPassenger.loyaltyCards as ITrainsLoyaltyCard[])
            .filter(({type}) => allowedLoyaltyCards.includes(type))
            .reduce<PassengerBonusCardsType>(
                (allBonusCards, {type, number}) => ({
                    ...allBonusCards,
                    [type]: {
                        value: number,
                    },
                }),
                {},
            ),
        gender: {
            value:
                travelerPassenger.gender.value === EGender.MALE
                    ? GENDER_TYPE.MALE
                    : GENDER_TYPE.FEMALE,
        },
        hasDiscountDocument: false,
        isNonRefundableTariff: false,
    };
}
