import moment, {Moment} from 'moment-timezone';

interface IStation {
    timezone: string;
}

interface ISegment {
    arrival: string;
    departure: string;
    stationFrom: IStation;
    stationTo: IStation;
}

const cache = new WeakMap<
    ISegment,
    PartialRecord<EField, PartialRecord<string, Moment>>
>();

enum EField {
    DEPARTURE = 'departure',
    ARRIVAL = 'arrival',
}

function getLocalTime(
    segment: ISegment,
    field: EField,
    timezone: string,
): Moment {
    const segmentCache = cache.get(segment);

    if (segmentCache) {
        const segmentCacheField = segmentCache[field];

        if (!segmentCacheField?.[timezone]) {
            segmentCache[field] = {
                ...segmentCacheField,
                [timezone]: moment.tz(segment[field], timezone),
            };
        }
    } else {
        cache.set(segment, {
            [field]: {
                [timezone]: moment.tz(segment[field], timezone),
            },
        });
    }

    return (
        cache.get(segment)?.[field]?.[timezone] ||
        moment.tz(segment[field], timezone)
    );
}

export function getDepartureTime(segment: ISegment, timezone: string): Moment {
    return getLocalTime(segment, EField.DEPARTURE, timezone);
}

export function getArrivalTime(segment: ISegment, timezone: string): Moment {
    return getLocalTime(segment, EField.ARRIVAL, timezone);
}

export function getLocalDepartureTime(segment: ISegment): Moment {
    return getDepartureTime(segment, segment.stationFrom.timezone);
}

export function getLocalArrivalTime(segment: ISegment): Moment {
    return getArrivalTime(segment, segment.stationTo.timezone);
}
