import {Moment} from 'moment-timezone';

import {
    getArrivalTime,
    getDepartureTime,
    getLocalArrivalTime,
    getLocalDepartureTime,
} from 'projects/trains/lib/segments/cachedSegmentTime';

interface IStation {
    timezone: string;
    railwayTimezone: string;
}

interface ISegment {
    arrival: string;
    departure: string;
    stationFrom: IStation;
    stationTo: IStation;
}

// Вернет true для пар Москва - Мытищи, Казань - Рязань, Астана - Саратов, Ташкент - Москва
// Вернет false для пар Москва - Екатеринбург, Омск - Тюмень, Актобе - Москва, Ташкент - Екатеринбург
function bothStationsHaveRailwayTimezone({
    departureMoment,
    departureRailwayMoment,
    arrivalMoment,
    arrivalRailwayMoment,
}: {
    departureMoment: Moment;
    departureRailwayMoment: Moment;
    arrivalMoment: Moment;
    arrivalRailwayMoment: Moment;
}): boolean {
    return (
        departureRailwayMoment &&
        arrivalRailwayMoment &&
        departureMoment.tz() === departureRailwayMoment.tz() &&
        arrivalMoment.tz() === arrivalRailwayMoment.tz()
    );
}

/**
 * Вернёт moment объекты с датой отправления/прибытия в местной и железнодорожной таймзонах
 *
 * @param  segment - данные сегмента
 */
export default function getSegmentTimes(segment: ISegment): {
    departureMoment: Moment;
    departureRailwayMoment: Moment | null;
    arrivalMoment: Moment;
    arrivalRailwayMoment: Moment | null;
} {
    const result = {
        departureMoment: getLocalDepartureTime(segment),
        departureRailwayMoment: getDepartureTime(
            segment,
            segment.stationFrom.railwayTimezone,
        ),
        arrivalMoment: getLocalArrivalTime(segment),
        arrivalRailwayMoment: getArrivalTime(
            segment,
            segment.stationTo.railwayTimezone,
        ),
    };

    if (bothStationsHaveRailwayTimezone(result)) {
        return {
            ...result,
            departureRailwayMoment: null,
            arrivalRailwayMoment: null,
        };
    }

    return result;
}
