import {IStationSettlement} from 'projects/trains/lib/segments/types';
import {ITrainsStation} from 'types/trains/common/station/ITrainsStation';

import {CHAR_NBSP, CHAR_RIGHT_ARROW} from 'utilities/strings/charCodes';

import * as i18nBlock from 'i18n/trains-segment-transfer';

function getShortTransferDescription(
    transferPoint: IStationSettlement | ITrainsStation,
): string {
    if (
        'titleLocative' in transferPoint &&
        transferPoint.titleLocative &&
        transferPoint.preposition
    ) {
        return i18nBlock.transferDashPointDashDescriptionDashShortDashLocative({
            transferPoint: transferPoint.titleLocative,
            preposition: transferPoint.preposition,
        });
    }

    return i18nBlock.transferDashPointDashDescriptionDashShortDashInfinitive({
        transferPoint: transferPoint.title,
    });
}

function getFullTransferDescription(
    settlement: IStationSettlement,
    stationTitle: string,
): string {
    if (settlement.titleLocative && settlement.preposition) {
        return i18nBlock.transferDashPointDashDescriptionDashFullDashLocative({
            settlement: settlement.titleLocative,
            settlementPreposition: settlement.preposition,
            station: stationTitle,
        });
    }

    return i18nBlock.transferDashPointDashDescriptionDashFullDashInfinitive({
        settlement: settlement.title,
        station: stationTitle,
    });
}

interface ISegment {
    stationTo: ITrainsStation;
    stationFrom: ITrainsStation;
}

/**
 * Вернёт описание пересадки
 *
 * @param segment - пересадочный сегмент
 * @param nextSegment - пересадочный сегмент
 */
export function getTransferPointDescription(
    segment: ISegment,
    nextSegment: ISegment,
): string {
    const {stationTo} = segment;
    const settlement = stationTo.settlement;

    if (settlement) {
        let station = stationTo.popularTitle || stationTo.title;
        const nextSegmentFrom = nextSegment.stationFrom;

        if (
            stationTo.id === nextSegmentFrom.id &&
            station.includes(settlement.title)
        ) {
            return getShortTransferDescription(settlement);
        }

        if (stationTo.id !== nextSegmentFrom.id) {
            const nextStation =
                nextSegmentFrom.popularTitle || nextSegmentFrom.title;

            station += `${CHAR_NBSP}${CHAR_RIGHT_ARROW} ${nextStation}`;
        }

        return getFullTransferDescription(settlement, station);
    }

    return getShortTransferDescription(stationTo);
}
