import {generatePath, matchPath} from 'react-router-dom';
import isEqual from 'lodash/isEqual';

import {ORDER_STEP} from 'projects/trains/constants/orderSteps';
import {URLs} from 'constants/urls';

import {isNotUndefined} from 'types/utilities';

import {TOrderStepDescription} from 'reducers/trains/order/thunk/changeOrderStep';

import isOrderStep from 'projects/trains/lib/order/steps/isOrderStep';
import {FIRST_PLACES_STEP} from 'projects/trains/lib/order/steps/firstPlacesStep';
import {
    convertDirectionToPathDirection,
    convertPathDirectionToDirection,
    TPathDirection,
} from 'projects/trains/lib/urls/pathDirectionHelpers';

const ORDER_STEP_PATH = {
    [ORDER_STEP.PLACES]: URLs.trainsOrderPlaces,
    [ORDER_STEP.PASSENGERS]: URLs.trainsOrderPassengers,
    [ORDER_STEP.CONFIRM]: URLs.trainsOrderConfirm,
    [ORDER_STEP.PAYMENT]: URLs.trainsOrderPayment,
};

/**
 * Возвращает путь соответствующий шагу покупки
 * @param orderDescription - шаг покупки
 * @returns
 */
export function getOrderStepUrl(
    orderDescription: TOrderStepDescription,
): string {
    if (orderDescription.step === ORDER_STEP.PLACES) {
        const path = ORDER_STEP_PATH[orderDescription.step];

        if (isEqual(orderDescription, FIRST_PLACES_STEP)) {
            return generatePath(path, {});
        }

        return generatePath(path, {
            direction: convertDirectionToPathDirection(
                orderDescription.direction,
            ),
            index: orderDescription.index,
        });
    }

    return ORDER_STEP_PATH[orderDescription.step];
}

export function getOrderStepDescriptionByPath(
    pathname: string,
): TOrderStepDescription {
    const matches: (TOrderStepDescription | undefined)[] = Object.entries(
        ORDER_STEP_PATH,
    ).map(([step, path]) => {
        if (!isOrderStep(step)) {
            return undefined;
        }

        const match = matchPath<{
            direction?: TPathDirection;
            index?: string;
        }>(pathname, {path, exact: true});

        if (!match) {
            return undefined;
        }

        if (step === ORDER_STEP.PLACES) {
            const {direction, index} = match.params;

            return {
                step,
                direction: convertPathDirectionToDirection(direction),
                index: index ? Number(index) : 0,
            };
        }

        return {step};
    });

    return matches.filter(isNotUndefined)[0] ?? FIRST_PLACES_STEP;
}
