import moment, {Moment} from 'moment-timezone';
import pick from 'lodash/pick';
import forEach from 'lodash/forEach';
import mapValues from 'lodash/mapValues';
import {ParsedQuery} from 'query-string';

import {
    PASSENGERS_TYPE_ORDER,
    PASSENGERS_TYPES,
} from 'projects/trains/constants/passengersTypes';
import {ORDER_STEP} from 'projects/trains/constants/orderSteps';
import {TRAIN_COACH_TYPE} from 'projects/trains/constants/coachType';
import {GENDER_TYPE} from 'projects/trains/constants/genders';
import {
    PAYMENT_TEST_CONTEXT_TOKEN_COOKIE_NAME,
    TRAIN_TEST_CONTEXT_TOKEN_COOKIE_NAME,
} from 'constants/testContext';

import {ITrainsFilledSearchContext} from 'reducers/trains/context/types';
import {TTrainsStoreOrderSegment} from 'projects/trains/lib/segments/types';
import {
    DIRECTIONS,
    DIRECTION_SHORT_NAMES,
    EDirection,
} from 'types/common/EDirection';
import {ITrainOrderState} from 'reducers/trains/order/types';

import {TrainsOrderStateInterface} from 'reducers/trains/order/reducer';
import {TOrderStepDescription} from 'reducers/trains/order/thunk/changeOrderStep';

import {ROBOT} from 'utilities/dateUtils/formats';
import {internalUrl} from 'utilities/url';
import getTrainNumberFromSegment from 'projects/trains/lib/segments/getTrainNumberFromSegment';
import checkSchemeHasGroupedPlaces from 'projects/trains/lib/order/coaches/checkSchemeHasGroupedPlaces';
import {getQueryByBrowserHistory} from 'utilities/getQueryByBrowserHistory/getQueryByBrowserHistory';

import getFirstForwardTrain from '../complexOrder/getFirstForwardTrain';
import {getOrderStepUrl} from './getOrderStepUrl';

interface IGetParametersFromSegment {
    number: string;
    fromId?: string;
    toId?: string;
    fromName?: string;
    toName?: string;
}

/**
 * Формирование GET параметров из сегмента
 *
 * @param segment - Сегмент.
 */
function getParametersFromSegment(
    segment: TTrainsStoreOrderSegment,
): IGetParametersFromSegment {
    const {stationFrom, stationTo} = segment;

    /**
     * Нужно, чтобы перезатереть параметры из контекста, т.к. в контексте будут лежать начальный и конечный пункты,
     * а не промежуточный с пересадкой.
     */
    const transferRouteParams =
        'isTransferSubSegment' in segment
            ? {
                  fromId: `s${stationFrom.id}`,
                  toId: `s${stationTo.id}`,
                  fromName: `s${stationFrom.title}`,
                  toName: `s${stationTo.title}`,
              }
            : {};

    return {
        number: getTrainNumberFromSegment(segment),
        ...transferRouteParams,
    };
}

interface IGetParametersFromContext {
    fromId: string;
    toId: string;
    fromName: string;
    toName: string;
    when: string;
    time: string;
    forwardSegmentId?: string | null;
    returnWhen: string | null;
}

/**
 * Формирование GET параметров из контекста.
 *
 * @param context - Поисковый контекст.
 * @param [departure] - Дата и время отправления.
 * @param [backwardDeparture] - Дата и время отправления обратно.
 */
function getParametersFromContext({
    context,
    departure,
    backwardDeparture,
}: {
    context: ITrainsFilledSearchContext;
    departure: Moment;
    backwardDeparture: Moment | null;
}): IGetParametersFromContext {
    const {from, to, forwardSegmentId} = context;

    return {
        fromId: from.key,
        toId: to.key,
        fromName: from.title,
        toName: to.title,
        when: departure && departure.format(ROBOT),
        returnWhen: backwardDeparture ? backwardDeparture.format(ROBOT) : null,
        forwardSegmentId,

        /*
         * Не HH:mm, чтобы избавиться от проблемы с двойным encode'ингом ':'. Точка не подвержена encode.
         *
         * @see https://st.yandex-team.ru/TRAVELFRONT-1096
         */
        time: departure && departure.format('HH.mm'),
    };
}

interface IGetSelectedParametersFromTrain {
    expandedServiceClassKey?: string;
    coachNumber?: string;
    coachType?: TRAIN_COACH_TYPE;
    gender?: GENDER_TYPE | null;
    place?: number[];
    petsAllowed?: boolean;
    bedding?: number;
    schemeHasGroupedPlaces?: boolean;
}

/**
 * Формирование выбранных юзером GET параметров одного поезда
 *
 * @param train - Информация о поезде
 */
function getSelectedParametersFromTrain(
    train?: ITrainOrderState | null,
): IGetSelectedParametersFromTrain {
    if (!train) {
        return {};
    }

    const coach = train.coach;
    const gender = train.gender;
    const orderPlaces = train.orderPlaces;
    const beddingOption = train.beddingOption;
    const expandedServiceClass = train.expandedServiceClass;
    const additionalSchemeInfo = train.additionalSchemeInfo;

    const data: IGetSelectedParametersFromTrain = {};

    if (expandedServiceClass) {
        data.expandedServiceClassKey = expandedServiceClass;
    }

    if (coach) {
        data.coachNumber = coach.number;
        data.coachType = coach.type;
        data.gender = gender;
        data.place = orderPlaces;

        if (coach.petsAllowed) {
            data.petsAllowed = coach.petsAllowed;
        }

        if (coach.canChooseBedding) {
            data.bedding = Number(beddingOption);
        }

        if (checkSchemeHasGroupedPlaces(coach, additionalSchemeInfo)) {
            data.schemeHasGroupedPlaces = true;
        }
    }

    return data;
}

type TGetSelectedParametersFromTrains = {
    [key in keyof IGetSelectedParametersFromTrain]?: string;
};

/**
 * Формирование выбранных юзером GET параметров всех поездов сложного заказа
 * Возвращает объект наподобие IGetParametersFromTrain, но
 * значения поездов префиксованы, склеены и разделены запятой
 *
 * @param order - Часть стора, соответствующая странице заказа.
 */
function getSelectedParametersFromTrains(
    order?: TrainsOrderStateInterface,
): IGetSelectedParametersFromTrain | TGetSelectedParametersFromTrains {
    if (!order) {
        return {};
    }

    if (
        Object.keys(order.trains[EDirection.BACKWARD]).length === 0 &&
        Object.keys(order.trains[EDirection.FORWARD]).length <= 1
    ) {
        return getSelectedParametersFromTrain(getFirstForwardTrain(order));
    }

    const trainsInfo: {
        [key in keyof TGetSelectedParametersFromTrains]?: string[];
    } = {};

    DIRECTIONS.forEach(direction => {
        forEach(order.trains[direction], (train, index) => {
            Object.entries(getSelectedParametersFromTrain(train)).forEach(
                ([key, value]) => {
                    const currentValues = (trainsInfo[
                        key as keyof IGetSelectedParametersFromTrain
                    ] =
                        trainsInfo[
                            key as keyof IGetSelectedParametersFromTrain
                        ] || []);

                    currentValues.push(
                        `${DIRECTION_SHORT_NAMES[direction]}${index}-${
                            Array.isArray(value) ? value.join('_') : value
                        }`,
                    );
                },
            );
        });
    });

    return mapValues(trainsInfo, values => values?.join(','));
}

interface IGetParametersFromOrder {
    [PASSENGERS_TYPES.ADULTS]?: number;
    [PASSENGERS_TYPES.CHILDREN]?: number;
    [PASSENGERS_TYPES.BABIES]?: number;
    id?: string;
    isTransfer?: '1';
    provider?: string;
}

/**
 * Формирование GET параметров из стора страницы покупки.
 *
 * @param order - Часть стора, соответствующая странице заказа.
 * @param orderStep - Шаг заказа.
 */
function getParametersFromOrder(
    order: TrainsOrderStateInterface,
    orderStep: ORDER_STEP,
): IGetParametersFromOrder {
    const {passengers, orderInfo} = order;

    const train = getFirstForwardTrain(order);
    const provider = train?.provider;

    const data: IGetParametersFromOrder = {};

    PASSENGERS_TYPE_ORDER.forEach(type => {
        if (passengers[type]) {
            data[type] = passengers[type];
        }
    });

    if (
        [ORDER_STEP.CONFIRM, ORDER_STEP.PAYMENT].includes(orderStep) &&
        orderInfo?.id
    ) {
        data.id = orderInfo.id;
    }

    if (provider) {
        data.provider = provider;
    }

    return data;
}

function getParametersFromHistory(): ParsedQuery {
    const query = getQueryByBrowserHistory();

    return pick(query, [
        'forward',
        'backward',
        TRAIN_TEST_CONTEXT_TOKEN_COOKIE_NAME,
        PAYMENT_TEST_CONTEXT_TOKEN_COOKIE_NAME,
    ]);
}

/**
 * Ссылка на покупку.
 *
 * @param params
 * @param [params.coachType] - Класс вагона (плацкарт, купе, сидячий...).
 * @param params.context - Поисковый контекст.
 * @param params.segment - Сегмент.
 * @param [params.order] - Часть стора, соответствующая странице заказа.
 * @param [params.departure] - Дата и время отправления. Если не задано, то время берется из segment
 * @param [params.orderStepDescription] - Шаг покупки
 *
 * @returns {String} - Ссылка.
 */
export function getTrainOrderUrl({
    coachType,
    context,
    segment,
    order,
    departure,
    orderStepDescription,
    backwardSegment,
}: {
    coachType?: TRAIN_COACH_TYPE;
    context: ITrainsFilledSearchContext;
    segment: TTrainsStoreOrderSegment;
    backwardSegment: TTrainsStoreOrderSegment | null;
    order?: TrainsOrderStateInterface;
    departure?: Moment;
    orderStepDescription: TOrderStepDescription;
}): string {
    const query = {
        coachType,
        ...getParametersFromContext({
            context,
            departure:
                departure ||
                moment.tz(segment.departure, segment.stationFrom.timezone),
            backwardDeparture: backwardSegment
                ? moment.tz(
                      backwardSegment.departure,
                      backwardSegment.stationFrom.timezone,
                  )
                : null,
        }),
        ...getParametersFromSegment(segment),
        ...(order
            ? getParametersFromOrder(order, orderStepDescription.step)
            : {}),
        ...getParametersFromHistory(),
        ...getSelectedParametersFromTrains(order),
    };

    return internalUrl(getOrderStepUrl(orderStepDescription), query, {
        filterNull: true,
    });
}
