import mapValues from 'lodash/mapValues';
import moment from 'moment-timezone';
import {ParsedQuery} from 'query-string';
import identity from 'lodash/identity';

import {PASSENGERS_TYPES} from 'projects/trains/constants/passengersTypes';
import {isTrainsCoachType} from 'projects/trains/constants/coachType';
import {isGenderType} from 'projects/trains/constants/genders';

import {
    ITrainRoutePart,
    ITrainsOrderLocation,
    TDirectionIndexQueryValue,
} from 'types/trains/booking/ITrainsOrderLocation';
import {
    DIRECTIONS_MAP_BY_SHORT_NAME,
    EDirection,
} from 'types/common/EDirection';
import {isNotUndefined} from 'types/utilities';

import {parseTrainRoutePart} from 'projects/trains/lib/urls/getTrainsOrderParamsByQuery/parseTrainRoutePart';
import boolify from 'utilities/boolean/boolify';
import {ROBOT} from 'utilities/dateUtils/formats';
import {getWizardReqIdFromQuery} from 'utilities/url/wizard/getWizardReqIdFromQuery';

export const SEGMENT_SEPARATOR = ',';

export function getTrainsOrderParamsByQuery(
    query: ParsedQuery,
): ITrainsOrderLocation {
    const {place} = query;
    const filteredQuery = mapValues(query, v => {
        if (Array.isArray(v)) {
            return v[0];
        }

        return v === null ? undefined : v;
    });

    const {
        id,
        forward,
        backward,
        place: places,
        gender,
        coachType,
        coachNumber,
        bedding,
        petsAllowed,
        expandedServiceClassKey,
        schemeHasGroupedPlaces,
        isTransfer,
        ['test-id']: testId,
        fromId,
        fromName,
        toId,
        toName,
        when,
        returnWhen,
        forwardSegmentId,
    } = filteredQuery;

    const wizardReqId = getWizardReqIdFromQuery(filteredQuery);

    const commonParams = {
        id,
        place: Array.isArray(place)
            ? {
                  [EDirection.FORWARD]: [
                      place.map(parseNumber).filter(isNotUndefined),
                  ],
              }
            : parseDirectionIndexQueryPart(places, places => {
                  if (!places) {
                      return [];
                  }

                  return places
                      .split('_')
                      .map(parseNumber)
                      .filter(isNotUndefined);
              }),
        gender: parseDirectionIndexQueryPart(gender, gender =>
            isGenderType(gender) ? gender : undefined,
        ),
        coachType: parseDirectionIndexQueryPart(coachType, coachType =>
            isTrainsCoachType(coachType) ? coachType : undefined,
        ),
        coachNumber: parseDirectionIndexQueryPart(coachNumber, identity),
        bedding: parseDirectionIndexQueryPart(bedding, parseBoolean),
        petsAllowed: parseDirectionIndexQueryPart(petsAllowed, parseBoolean),
        expandedServiceClassKey: parseDirectionIndexQueryPart(
            expandedServiceClassKey,
            identity,
        ),
        schemeHasGroupedPlaces: parseDirectionIndexQueryPart(
            schemeHasGroupedPlaces,
            parseBoolean,
        ),
        isTransfer: parseBoolean(isTransfer),
        wizardReqId,
        ['test-id']: testId,
        ...Object.values(PASSENGERS_TYPES).reduce((acc, type) => {
            const value = filteredQuery[type];

            return {
                ...acc,
                [type]: parseNumber(value),
            };
        }, {}),
        fromId,
        fromName,
        toId,
        toName,
        when,
        returnWhen,
        forwardSegmentId,
    };

    if (forward) {
        return {
            ...commonParams,
            [EDirection.FORWARD]: parseDirectionString(forward),
            [EDirection.BACKWARD]: parseDirectionString(backward),
        };
    }

    const {provider = 'P1', time, number} = filteredQuery;

    if (!fromId || !toId || !number || !when || !time) {
        return commonParams;
    }

    return {
        ...commonParams,
        [EDirection.FORWARD]: [
            {
                provider,
                number,
                fromId,
                toId,
                when: moment(`${when}T${time}`, `${ROBOT}THH.mm`),
            },
        ],
    };
}

/**
 * Парсит query-параметр с данными нескольких поездов
 *
 * Для строки "f0-<data0>,f1-<data1>"
 * вернет {FORWARD: {0: parser(<data0>), 1: parser(<data1>)}}
 */
function parseDirectionIndexQueryPart<T>(
    queryPart: string | undefined,
    parser: (value: string) => T,
): TDirectionIndexQueryValue<T> | undefined {
    if (!queryPart) {
        return;
    }

    return queryPart
        .split(',')
        .reduce<TDirectionIndexQueryValue<T>>((directionIndexValue, value) => {
            /**
             * Если есть совпадение, то это новая форма записи из сложного заказа
             * иначе фолбэк на старый вариант с одним поездом
             *
             * f/b - короткие названия направлений из DIRECTION_SHORT_NAMES
             */
            const match = value.match(/^([fb])(\d)-(.*)$/) || [
                '',
                'f',
                '0',
                value,
            ];
            const direction = DIRECTIONS_MAP_BY_SHORT_NAME[match[1]];
            const index = Number(match[2]);

            return {
                ...directionIndexValue,
                [direction]: {
                    ...directionIndexValue[direction],
                    [index]: parser(match[3]),
                },
            };
        }, {});
}

function parseDirectionString(
    direction: string | undefined,
): ITrainRoutePart[] | undefined {
    if (!direction) {
        return undefined;
    }

    return direction.split(SEGMENT_SEPARATOR).map(parseTrainRoutePart);
}

function parseBoolean(queryBoolean: string | undefined): boolean | undefined {
    return queryBoolean === undefined ? undefined : boolify(queryBoolean);
}

function parseNumber(queryNumber: string | undefined): number | undefined {
    if (queryNumber === undefined || isNaN(Number(queryNumber))) {
        return undefined;
    }

    return Number(queryNumber);
}
