import moment from 'moment-timezone';

import {URLs} from 'constants/urls';

import {ITrainsVariant} from 'types/trains/common/variant/ITrainsVariant';
import {ITrainsFilledSearchContext} from 'reducers/trains/context/types';
import {ITrainsSegment} from 'types/trains/common/segment/ITrainsSegment';
import {ITrainsMinTariffClass} from 'types/trains/common/tariffs/ITrainsMinTariffsClass';

import {internalUrl} from 'utilities/url';

import {SEGMENT_SEPARATOR} from './getTrainsOrderParamsByQuery/getTrainsOrderParamsByQuery';
import {stringifyTrainRoutePart} from './getTrainsOrderParamsByQuery/parseTrainRoutePart';

export const convertTrainsSegmentOptionsToString = (
    segment: ITrainsSegment,
): string => {
    const {provider, departure, stationFrom, stationTo, train} = segment;

    if (!provider) {
        return '';
    }

    return stringifyTrainRoutePart({
        provider,
        toId: String(stationTo.id),
        number: train.number,
        fromId: String(stationFrom.id),
        when: moment.tz(departure, stationFrom.timezone),
    });
};

const prepareAndConvertTrainsSegmentsToString = (
    segments: ITrainsSegment[],
): string | null => {
    if (!segments.length) {
        return null;
    }

    return segments
        .map(convertTrainsSegmentOptionsToString)
        .filter(Boolean)
        .join(SEGMENT_SEPARATOR);
};

export const getTrainsOrderUrlBySearchParams = ({
    variant,
    context,
    tariff,
}: {
    variant: ITrainsVariant;
    context: ITrainsFilledSearchContext;
    tariff?: ITrainsMinTariffClass;
}): string => {
    const {from, to, when, returnWhen, forwardSegmentId} = context;
    const {forward, backward} = variant;

    return internalUrl(
        `${URLs.trainsOrder}`,
        {
            when,
            fromId: from.key,
            toId: to.key,
            coachType: tariff?.type,
            returnWhen: returnWhen ?? null,
            forwardSegmentId: forwardSegmentId ?? null,
            forward: prepareAndConvertTrainsSegmentsToString(forward),
            backward: prepareAndConvertTrainsSegmentsToString(backward),
        },
        {filterNull: true},
    );
};
